import get from 'lodash/get';
import flatMap from 'lodash/flatMap';

import {
    IBookingFlightModel,
    IBookingSegmentModel,
} from 'server/services/AviaBookingService/variants/types';
import {ITermValue} from 'server/api/AviaBookingApi/types/ITermValue';
import {
    IFlightTariff,
    IFlightTariffs,
    IVariantForFlightTariff,
} from '../types/tariffTypes';

import * as localized from 'i18n/avia-AviaBooking-TariffsModal';

type TFareTermComparator = [string, (a: ITermValue, b: ITermValue) => boolean];

const comparators: TFareTermComparator[] = [
    ['refundable', (a, b): boolean => a.availability === b.availability],
    ['changingCarriage', (a, b): boolean => a.availability === b.availability],
    ['carryOn', (a, b): boolean => a.places === b.places && a.size === b.size],
    ['baggage', (a, b): boolean => a.places === b.places && a.size === b.size],
    ['miles', (a, b): boolean => a.miles === b.miles],
];

function areFareTermsEqual(
    fareTermsA: Record<string, ITermValue>,
    fareTermsB: Record<string, ITermValue>,
): boolean {
    return comparators.every(([key, comparator]) => {
        const termA = fareTermsA[key];
        const termB = fareTermsB[key];

        if (termA && termB) {
            return comparator(termA, termB);
        }

        return false;
    });
}

function areFlightsHaveEqualFareTerms([
    firstFlight,
    ...restFlights
]: IBookingFlightModel[]): boolean {
    if (!firstFlight) {
        return true;
    }

    let flightsAreEqual = true;
    let flight;

    while (flightsAreEqual && (flight = restFlights.pop())) {
        flightsAreEqual = areFareTermsEqual(
            firstFlight.fareTerms.terms,
            flight.fareTerms.terms,
        );
    }

    return flightsAreEqual;
}

function areSegmentsHaveEqualFareTerms(
    segments: IBookingSegmentModel[],
): boolean {
    return areFlightsHaveEqualFareTerms(flatMap(segments, x => x.flights));
}

function areSegmentsForEachVariantHaveEqualFareTerms(
    variants: IVariantForFlightTariff[],
): boolean {
    return variants.every(variant =>
        areSegmentsHaveEqualFareTerms(variant.segments),
    );
}

export function collectFlightTariffsFromVariants(
    variants: IVariantForFlightTariff[],
): Map<string, IFlightTariffs> {
    const flightTariffs = new Map<string, IFlightTariffs>();

    function setFlightTariff(
        flight: IBookingFlightModel,
        variant: IVariantForFlightTariff,
    ): void {
        if (!flightTariffs.has(flight.id)) {
            flightTariffs.set(flight.id, {
                title: `${flight.departureSettlement.phraseFrom} — ${flight.arrivalSettlement.phraseTo}`,
                tariffs: new Map<string, IFlightTariff>(),
            });
        }

        const tariffs = flightTariffs.get(flight.id)?.tariffs;

        tariffs?.set(variant.id, {
            priceInfo: variant.priceInfo,
            promoCampaigns: variant.promoCampaigns,
            fareTerms: flight.fareTerms,
        });
    }

    if (areSegmentsForEachVariantHaveEqualFareTerms(variants)) {
        const {id} = get(variants, '[0].segments[0].flights[0]');

        flightTariffs.set(id, {
            title:
                variants.length > 1
                    ? localized.allDashTariffsDashTitle()
                    : localized.oneTariffTitle(),
            tariffs: new Map<string, IFlightTariff>(),
        });

        for (const variant of variants) {
            const segment = variant.segments[0];
            const flight = segment.flights[0];

            setFlightTariff(flight, variant);
        }

        return flightTariffs;
    }

    for (const variant of variants) {
        for (const segment of variant.segments) {
            for (const flight of segment.flights) {
                setFlightTariff(flight, variant);
            }
        }
    }

    return flightTariffs;
}
