import React, {createContext, useContext, useMemo} from 'react';
import {noop} from 'lodash';

import {useToggle} from 'utilities/hooks/useToggle';
import {useBoolean} from 'utilities/hooks/useBoolean';

export interface IAviaUiDynamicsContext {
    dynamicsIsVisible: boolean;
    toggleDynamics: () => void;
    overlayIsVisible: boolean;
    showOverlay: () => void;
    hideOverlay: () => void;
}

export type TAviaUiContext = IAviaUiDynamicsContext;

export const AviaUiContext = createContext<TAviaUiContext>({
    dynamicsIsVisible: false,
    toggleDynamics: noop,
    overlayIsVisible: false,
    showOverlay: noop,
    hideOverlay: noop,
});

export function useAviaUiContext(): TAviaUiContext {
    return useContext(AviaUiContext);
}

export const AviaUiProvider: React.FC = ({children}) => {
    const [dynamicsIsVisible, toggleDynamics] = useToggle(false);
    const {
        value: overlayIsVisible,
        setTrue: showOverlay,
        setFalse: hideOverlay,
    } = useBoolean(false);
    const uiContext = useMemo<TAviaUiContext>(
        () => ({
            dynamicsIsVisible,
            toggleDynamics,
            overlayIsVisible,
            showOverlay,
            hideOverlay,
        }),
        [
            dynamicsIsVisible,
            toggleDynamics,
            overlayIsVisible,
            showOverlay,
            hideOverlay,
        ],
    );

    return (
        <AviaUiContext.Provider value={uiContext}>
            {children}
        </AviaUiContext.Provider>
    );
};
