import {useDispatch, useSelector} from 'react-redux';
import {useCallback, useEffect, useMemo, useRef} from 'react';
import {isEqual} from 'lodash';

import {AVIA_SORTING_FILTER_ITEMS} from 'projects/avia/components/AviaSearchResultsSortingFilter/constants';

import {IAviaSearchResultsSorting} from 'reducers/avia/search/results/reducer';
import {setResultsSorting} from 'reducers/avia/search/results/actions';
import {EAviaResultsSortType} from 'reducers/avia/search/results/EAviaResultsSortType';

import {getAviaSearchResults} from 'selectors/avia/search/aviaSearchResultsSelector';

import {EAviaRumEvents} from 'projects/avia/lib/EAviaRumEvents';
import {
    getSortingFromCookie,
    setSortingCookie,
} from 'projects/avia/lib/search/aviaSortingCookie';
import {ESortDirection} from 'projects/avia/lib/comparators/TComparator';
import {useExperiments} from 'utilities/hooks/useExperiments';

import {useRumUi} from 'contexts/RumUiContext';

type TAviaSortingFilterHandler = (payload: IAviaSearchResultsSorting) => void;

export interface IUseAviaSortingFilter {
    sorting: IAviaSearchResultsSorting;
    handleSortChange: TAviaSortingFilterHandler;
    isActive: boolean;
    sortingName?: string;
}

export const useAviaSortingFilter = (): IUseAviaSortingFilter => {
    const {sorting} = useSelector(getAviaSearchResults);
    const dispatch = useDispatch();
    const rumUi = useRumUi();
    const sortingRef = useRef(sorting);

    const {aviaDefaultSortPrice} = useExperiments();

    /*
     * при выпиливании экспа aviaDefaultSortPrice это стоит вынести в
     * константы в одно место и её использовать здесь и как дефолтное значение
     * в сторе
     */
    const defaultSorting: IAviaSearchResultsSorting = useMemo(
        () =>
            aviaDefaultSortPrice
                ? {
                      direction: ESortDirection.ASCENDING,
                      type: EAviaResultsSortType.PRICE,
                  }
                : {
                      direction: ESortDirection.ASCENDING,
                      type: EAviaResultsSortType.INTEREST,
                  },
        [aviaDefaultSortPrice],
    );

    const isActive = useMemo(
        () => !isEqual(defaultSorting, sorting),
        [sorting, defaultSorting],
    );

    const sortingName = useMemo(() => {
        const {type, direction} = sorting;

        return AVIA_SORTING_FILTER_ITEMS.find(
            item => item.type === type && item.direction === direction,
        )?.label;
    }, [sorting]);

    const handleSortChange = useCallback<TAviaSortingFilterHandler>(
        payload => {
            rumUi.measure(EAviaRumEvents.Sort);
            dispatch(setResultsSorting(payload));
        },
        [dispatch, rumUi],
    );

    useEffect(() => {
        const sortingFromCookie = getSortingFromCookie();

        if (
            sortingFromCookie &&
            !isEqual(sortingRef.current, sortingFromCookie)
        ) {
            handleSortChange(sortingFromCookie);
        }

        // должно отработать только при первом рендере
    }, []);

    useEffect(() => {
        const sortingFromCookie = getSortingFromCookie();

        /*
         * костыль под эксперимент aviaDefaultSortPrice, т.к при нем дефолтное
         * альтернативное значение устанавливается уже после инициализации
         * приложения, не хотим в этом кейсе её запоминать в куку
         */
        if (
            !sortingFromCookie &&
            aviaDefaultSortPrice &&
            isEqual(defaultSorting, sorting)
        ) {
            return;
        }

        setSortingCookie(sorting);

        // должно выполняться только при изменении сортировки
    }, [sorting]);

    return {sorting, handleSortChange, isActive, sortingName};
};
