import {IBookingFormData} from 'projects/avia/pages/AviaBooking/types/IBookingFormData';
import {IBookingCreateOrderRequest} from 'projects/avia/pages/AviaBooking/types/IBookingCreateOrder';
import {IBookingFormPassengersData} from 'projects/avia/pages/AviaBooking/types/IBookingFormPassengersData';
import {IAviaCreateOrderDocumentDTO} from 'server/api/AviaBookingApi/types/ICreateOrderRequest';
import {TAviaFormDocumentTypes} from 'server/api/AviaBookingApi/types/TAviaFormDocumentTypes';

import {parseDate} from 'utilities/dateUtils';
import {ROBOT, HUMAN_DATE_RU} from 'utilities/dateUtils/formats';

import {TBookingContacts} from 'components/BookingContactsForm/BookingContactsForm';

import {EDocumentType} from 'server/api/TravelersApi/enums/EDocumentType';
import {AviaDocumentType} from 'server/api/AviaBookingApi/enums/DocumentType';

const MAP_COMMON_DOCUMENT_TYPE_TO_API: Record<
    TAviaFormDocumentTypes,
    AviaDocumentType
> = {
    [EDocumentType.RU_NATIONAL_PASSPORT]: AviaDocumentType.PASSPORT,
    [EDocumentType.RU_FOREIGN_PASSPORT]:
        AviaDocumentType.INTERNATIONAL_PASSPORT_RU,
    [EDocumentType.RU_BIRTH_CERTIFICATE]: AviaDocumentType.BIRTH_CERTIFICATE,
    [EDocumentType.OTHER]: AviaDocumentType.NATIONAL_PASSPORT,
};

function prepareFormDocumentToRequest(
    document: IBookingFormPassengersData,
): IAviaCreateOrderDocumentDTO | null {
    if (
        document.firstName &&
        document.lastName &&
        document.birthdate &&
        document.citizenship &&
        document.documentType &&
        document.documentNumber
    ) {
        const isCardAndNumberFilled =
            document.loyaltyCard?.type && document.loyaltyCard?.number;

        return {
            firstName: document.firstName,
            lastName: document.lastName,
            middleName: document.patronymicName,
            sex: document.sex,
            dateOfBirth: parseDate(document.birthdate, HUMAN_DATE_RU).format(
                ROBOT,
            ),
            citizenship: document.citizenship,
            documentType:
                MAP_COMMON_DOCUMENT_TYPE_TO_API[document.documentType],
            documentNumber: document.documentNumber,
            documentValidTill: document.documentValidDate
                ? parseDate(document.documentValidDate, HUMAN_DATE_RU).format(
                      ROBOT,
                  )
                : '',
            loyaltyProgramInternalCode: isCardAndNumberFilled
                ? document.loyaltyCard?.type
                : undefined,
            loyaltyProgramAccountNumber: isCardAndNumberFilled
                ? document.loyaltyCard?.number
                : undefined,
        };
    }

    return null;
}

function isApiValidDocuments(
    documents: (IAviaCreateOrderDocumentDTO | null)[],
): documents is IAviaCreateOrderDocumentDTO[] {
    return documents.every(
        (document): document is IAviaCreateOrderDocumentDTO =>
            Boolean(document),
    );
}

function isApiValidContacts(
    contacts: TBookingContacts,
): contacts is Required<TBookingContacts> {
    return Boolean(contacts.email) && Boolean(contacts.phone);
}

export function prepareFormDataToRequest(
    formData: IBookingFormData,
): IBookingCreateOrderRequest | null {
    const {flightInfo, passengers = [], contacts = {}, marker} = formData;
    const requestDocuments = passengers.map(prepareFormDocumentToRequest);

    if (isApiValidDocuments(requestDocuments) && isApiValidContacts(contacts)) {
        return {
            documents: requestDocuments,
            email: contacts.email,
            phone: contacts.phone,
            variantToken: flightInfo?.variantToken || '',
            marker,
        };
    }

    return null;
}
