import moment from 'moment';

import TDateRobot from 'types/common/date/TDateRobot';

import getDateRobotFromMoment from 'utilities/dateUtils/getDateRobotFromMoment';

export interface IGetIntervalsForRequestTableDynamicCoords {
    // Количество дней слева от текущей даты "туда-обратно", данные которых нужно запросить
    leftRange: number;

    // Количество дней справа от текущей даты "туда-обратно", данные которых нужно запросить
    rightRange?: number;
    // Количество дней сверху от текущей даты "туда-обратно", данные которых нужно запросить
    topRange?: number;
    // Количество дней снизу от текущей даты "туда-обратно", данные которых нужно запросить
    bottomRange?: number;
}

export interface IGetIntervalsForRequestTableDynamicProps
    extends IGetIntervalsForRequestTableDynamicCoords {
    forwardDate: TDateRobot;
    backwardDate: TDateRobot;
}

export interface ITableInterval {
    when: TDateRobot;
    returnDate: TDateRobot;
    startDate: TDateRobot;
    endDate: TDateRobot;
}

export interface IGetIntervalsForRequestTableDynamic {
    (params: IGetIntervalsForRequestTableDynamicProps): ITableInterval[];
}

/**
 * Возвращает набор данных для подстановки в параметры запроса динамики цен для
 * табличного представления.
 */
const getIntervalsForRequestTableDynamic: IGetIntervalsForRequestTableDynamic =
    ({
        forwardDate,
        backwardDate,
        leftRange,
        rightRange = leftRange,
        topRange = leftRange,
        bottomRange = topRange,
    }) => {
        const forwardMoment = moment(forwardDate).add(-topRange, 'days');
        const forwardMomentMax = moment(forwardDate).add(bottomRange, 'days');
        const backwardMoment = moment(backwardDate).add(-leftRange, 'days');
        const backwardMomentMax = moment(backwardDate).add(rightRange, 'days');

        // Формируем массивы с возможными датами туда и обратно
        const forwardDates: TDateRobot[] = [];

        while (forwardMoment.isSameOrBefore(forwardMomentMax)) {
            forwardDates.push(getDateRobotFromMoment(forwardMoment));
            forwardMoment.add(1, 'day');
        }

        const backwardDates: TDateRobot[] = [];

        while (backwardMoment.isSameOrBefore(backwardMomentMax)) {
            backwardDates.push(getDateRobotFromMoment(backwardMoment));
            backwardMoment.add(1, 'day');
        }

        const intervalsData: ITableInterval[] = [];

        // Сначала пройдем все диагонали в правой верхней части таблицы.
        // Иными словами будем увеличивать даты "обратно" для минимальной даты "туда".
        for (
            let backwardDateIndex = 0;
            backwardDateIndex < backwardDates.length;
            backwardDateIndex++
        ) {
            intervalsData.push(
                getInfoForStartParameters({
                    forwardDateIndex: 0,
                    backwardDateIndex,
                    forwardDates,
                    backwardDates,
                }),
            );
        }

        // Далее пройдем все диагонали в левой нижней части таблицы.
        // Иными словами будем увеличивать даты "туда" для минимальной даты обратно
        for (
            let forwardDateIndex = 1;
            forwardDateIndex < forwardDates.length;
            forwardDateIndex++
        ) {
            intervalsData.push(
                getInfoForStartParameters({
                    forwardDateIndex,
                    backwardDateIndex: 0,
                    forwardDates,
                    backwardDates,
                }),
            );
        }

        // Профильтруем невалидные данные для запросов и вернем результат
        return intervalsData.filter(({when, returnDate}) => when <= returnDate);
    };

export default getIntervalsForRequestTableDynamic;

interface IGetInfoForStartParameters {
    forwardDateIndex: number;
    backwardDateIndex: number;
    forwardDates: TDateRobot[];
    backwardDates: TDateRobot[];
}

/**
 * Вычисляет параметры запроса для диагонали дат
 *
 * Параметры forwardDateIndex и backwardDateIndex - это "координаты" старта диагонали по
 * левой стороне (forwardDateIndex) и верхней стороне (backwardDateIndex)
 */
function getInfoForStartParameters({
    forwardDateIndex,
    backwardDateIndex,
    forwardDates,
    backwardDates,
}: IGetInfoForStartParameters): ITableInterval {
    const maxForwardIndex = forwardDates.length - 1;
    const maxBackwardIndex = backwardDates.length - 1;
    const dayDeltaForward = maxForwardIndex - forwardDateIndex;
    const dayDeltaBackward = maxBackwardIndex - backwardDateIndex;
    const dayDelta = Math.min(dayDeltaForward, dayDeltaBackward);

    return {
        when: forwardDates[forwardDateIndex],
        returnDate: backwardDates[backwardDateIndex],
        startDate: forwardDates[forwardDateIndex],
        endDate: forwardDates[forwardDateIndex + dayDelta],
    };
}
