import {TCountryKey, TSettlementKey, TStationKey} from 'types/PointKey';
import {EAviaSuggestsApiItemType} from 'server/api/AviaSuggestsApi/types/TAviaGetSuggestsApiResponse';
import {IAviaGeoService} from 'server/services/AviaGeoService/types/IAviaGeoService';

import {IPreparedAviaSuggestItemWithIsUniqueTitle} from 'server/services/AviaService/utilities/prepareSuggestsResponse';
import {isStationKey} from 'utilities/strings/isStationKey';
import {isSettlementKey} from 'utilities/strings/isSettlementKey';
import {getIdFromPointKey} from 'utilities/strings/getIdFromPointKey';

/**
 * Возвращает структуру саджеста поисковой формы авиа для пунктов откуда/куда по
 * pointKey данного пункта.
 * В зависимости от того, что это за тип pointKey пойдет в разные ручки.
 */
export async function getSuggestByPointKey(
    aviaGeoService: IAviaGeoService,
    pointKey: TSettlementKey | TStationKey | TCountryKey,
): Promise<IPreparedAviaSuggestItemWithIsUniqueTitle | null> {
    // пункт - город или станция
    if (isStationKey(pointKey) || isSettlementKey(pointKey)) {
        const geoPoint = await aviaGeoService.getDataByPointKey(pointKey);

        if (!geoPoint) {
            return null;
        }

        const {station, settlement, region, country} = geoPoint;

        if (!settlement || !country) {
            return null;
        }

        // Подгоняем ответ ручки geo/point под формат саджестов
        return {
            isUniqueTitle: false,
            type: station
                ? EAviaSuggestsApiItemType.AIRPORT
                : EAviaSuggestsApiItemType.CITY,
            title: station?.title || settlement.title,
            pointCode: station?.iataCode || settlement.iata || '',
            pointKey,
            countryTitle: country.title,
            cityTitle: settlement.title,
            regionTitle: region?.title || '',
        };
    }

    // Пункт - страна
    const country = await aviaGeoService.getCountry(
        getIdFromPointKey(pointKey),
    );

    if (!country) {
        return null;
    }

    return {
        isUniqueTitle: false,
        type: EAviaSuggestsApiItemType.COUNTRY,
        title: country.title,
        pointCode: country.code,
        pointKey: country.point_key,
        countryTitle: country.title,
        cityTitle: '',
        regionTitle: '',
    };
}
