import {createSelector} from 'reselect';
import {useCallback} from 'react';

import {
    setDirectionAirportsFilter,
    setTransferAirportsFilter,
    resetDirectionAirportsFilter,
    resetTransferAirportsFilter,
    resetFilterValue,
} from 'reducers/avia/search/results/filters/actions';
import {
    IAviaSearchSegmentsDirectionAirportsFilter,
    IAviaSearchSegmentsTransferAirportsFilter,
} from 'reducers/avia/search/results/filters/reducer';

import {aviaFiltersViewDataSelector} from 'selectors/avia/search/filters/viewData';
import {aviaAirportsFilterSelector} from 'selectors/avia/search/filters/airportsFilter';

import {useAsyncAction} from 'utilities/hooks/useAsyncAction';
import {useAviaFilteringFlow} from 'projects/avia/lib/hooks/useFilteringFlow';

const filterSelector = createSelector(
    aviaAirportsFilterSelector.value,
    aviaFiltersViewDataSelector.airports,
    aviaAirportsFilterSelector.directionsSelectedCount,
    aviaAirportsFilterSelector.directionsIsActive,
    aviaAirportsFilterSelector.transfersSelectedCount,
    aviaAirportsFilterSelector.transfersIsActive,
    (
        value,
        viewData,
        directionsCount,
        directionsIsActive,
        transfersCount,
        transfersIsActive,
    ) => ({
        value,
        viewData,
        directionsCount,
        directionsIsActive,
        transfersCount,
        transfersIsActive,
    }),
);

export interface IUseAviaAirportsFilter {
    filter: ReturnType<typeof filterSelector>;
    handleDirectionAirports: (
        value: IAviaSearchSegmentsDirectionAirportsFilter[],
    ) => void;
    handleTransferAirports: (
        value: IAviaSearchSegmentsTransferAirportsFilter[],
    ) => void;
    resetDirectionAirports: () => void;
    resetTransferAirports: () => void;
    reset: () => void;
}

export const useAviaAirportsFilter = (): IUseAviaAirportsFilter => {
    const filtering = useAviaFilteringFlow();
    const [filter, handleAsync] = useAsyncAction(filterSelector, filtering.end);
    const handleDirectionAirports = useCallback(
        (value: IAviaSearchSegmentsDirectionAirportsFilter[]) => {
            filtering.start(setDirectionAirportsFilter(value));
            handleAsync(setDirectionAirportsFilter, value);
        },
        [handleAsync, filtering],
    );
    const handleTransferAirports = useCallback(
        (value: IAviaSearchSegmentsTransferAirportsFilter[]) => {
            filtering.start(setTransferAirportsFilter(value));
            handleAsync(setTransferAirportsFilter, value);
        },
        [handleAsync, filtering],
    );
    const resetDirectionAirports = useCallback(() => {
        filtering.start(resetDirectionAirportsFilter());
        handleAsync(resetDirectionAirportsFilter, undefined);
    }, [handleAsync, filtering]);
    const resetTransferAirports = useCallback(() => {
        filtering.start(resetTransferAirportsFilter());
        handleAsync(resetTransferAirportsFilter, undefined);
    }, [handleAsync, filtering]);
    const reset = useCallback(() => {
        filtering.start(resetFilterValue('airports'));
        handleAsync(resetFilterValue, 'airports');
    }, [handleAsync, filtering]);

    return {
        filter,
        handleDirectionAirports,
        handleTransferAirports,
        resetDirectionAirports,
        resetTransferAirports,
        reset,
    };
};
