import {useCallback} from 'react';
import {createSelector} from 'reselect';

import {setTransferFilter} from 'reducers/avia/search/results/filters/actions';
import {EAviaSearchTransferFilter} from 'reducers/avia/search/results/filters/reducer';

import {aviaTransferFilterSelector} from 'selectors/avia/search/filters/transferFilter';
import {aviaFiltersViewDataSelector} from 'selectors/avia/search/filters/viewData';

import {disable, enable, isEnabled} from 'utilities/flags/flags';
import {useAsyncAction} from 'utilities/hooks/useAsyncAction';
import {useAviaFilteringFlow} from 'projects/avia/lib/hooks/useFilteringFlow';

const filterSelector = createSelector(
    aviaTransferFilterSelector.value,
    aviaTransferFilterSelector.isActive,
    aviaFiltersViewDataSelector.noTransferMinPrice,
    aviaTransferFilterSelector.selectedCount,
    (value, isActive, price, count) => ({
        value,
        isActive,
        price,
        count,
    }),
);

export interface IUseAviaNoTransfersFilter {
    filter: ReturnType<typeof filterSelector>;
    handle: () => void;
    reset: () => void;
}

export const useAviaNoTransfersFilter = (): IUseAviaNoTransfersFilter => {
    const filtering = useAviaFilteringFlow();
    const [filter, handleAsync] = useAsyncAction(
        filterSelector,
        filtering.end,
        200,
    );
    const handle = useCallback(() => {
        const {value: flags} = filter.value;
        const value = isEnabled(flags, EAviaSearchTransferFilter.NO_TRANSFERS)
            ? disable(flags, EAviaSearchTransferFilter.NO_TRANSFERS)
            : enable(flags, EAviaSearchTransferFilter.NO_TRANSFERS);

        filtering.start(setTransferFilter(value));
        handleAsync(setTransferFilter, value);
    }, [filter.value, handleAsync, filtering]);
    const reset = useCallback(() => {
        const {value: flags} = filter.value;
        const value = disable(flags, EAviaSearchTransferFilter.NO_TRANSFERS);

        filtering.start(setTransferFilter(value));
        handleAsync(setTransferFilter, value);
    }, [handleAsync, filtering, filter.value]);

    return {filter, handle, reset};
};
