import {useCallback} from 'react';
import {createSelector} from 'reselect';

import {
    resetFilterValue,
    setTimeFilter,
} from 'reducers/avia/search/results/filters/actions';
import {IAviaSearchTimeFilter} from 'reducers/avia/search/results/filters/reducer';

import {aviaFiltersViewDataSelector} from 'selectors/avia/search/filters/viewData';
import {aviaTimeFilterSelector} from 'selectors/avia/search/filters/timeFilter';

import {useAsyncAction} from 'utilities/hooks/useAsyncAction';
import {useAviaFilteringFlow} from 'projects/avia/lib/hooks/useFilteringFlow';

const filterSelector = createSelector(
    aviaTimeFilterSelector.value,
    aviaFiltersViewDataSelector.time,
    aviaTimeFilterSelector.selectedCount,
    aviaTimeFilterSelector.isActive,
    (value, viewData, count, isActive) => ({
        value,
        viewData,
        count,
        isActive,
    }),
);

export interface IUseAviaTimeFilter {
    filter: ReturnType<typeof filterSelector>;
    handle: (segmentVal: IAviaSearchTimeFilter, segmentIdx: number) => void;
    reset: () => void;
}

export function useAviaTimeFilter(): IUseAviaTimeFilter {
    const filtering = useAviaFilteringFlow();
    const [filter, handleAsync] = useAsyncAction(filterSelector, filtering.end);
    const handle = useCallback(
        (segmentVal: IAviaSearchTimeFilter, segmentIdx: number) => {
            const nextFilterValue = filter.value.slice();

            nextFilterValue[segmentIdx] = segmentVal;

            filtering.start(setTimeFilter(nextFilterValue));
            handleAsync(setTimeFilter, nextFilterValue);
        },
        [filter.value, handleAsync, filtering],
    );
    const reset = useCallback(() => {
        filtering.start(resetFilterValue('time'));
        handleAsync(resetFilterValue, 'time');
    }, [handleAsync, filtering]);

    return {filter, handle, reset};
}
