import {useCallback} from 'react';
import {createSelector} from 'reselect';

import {EAviaSearchTransferFilter} from 'reducers/avia/search/results/filters/reducer';
import {
    resetFilterValue,
    setTransferFilter,
} from 'reducers/avia/search/results/filters/actions';

import {aviaTransferFilterSelector} from 'selectors/avia/search/filters/transferFilter';
import {aviaFiltersViewDataSelector} from 'selectors/avia/search/filters/viewData';

import {Flag} from 'utilities/flags/flags';
import {useAsyncAction} from 'utilities/hooks/useAsyncAction';
import {useAviaFilteringFlow} from 'projects/avia/lib/hooks/useFilteringFlow';

const filterSelector = createSelector(
    aviaTransferFilterSelector.value,
    aviaFiltersViewDataSelector.transfer,
    aviaTransferFilterSelector.selectedCount,
    aviaTransferFilterSelector.isActive,
    aviaFiltersViewDataSelector.noTransferMinPrice,
    (value, viewData, count, isActive, price) => ({
        value,
        viewData,
        count,
        isActive,
        price,
    }),
);

export interface IUseAviaTransfersFilter {
    filter: ReturnType<typeof filterSelector>;
    handle: (value: Flag<EAviaSearchTransferFilter>) => void;
    reset: () => void;
}

export function useAviaTransfersFilter(): IUseAviaTransfersFilter {
    const filtering = useAviaFilteringFlow();
    const [filter, handleAsync] = useAsyncAction(filterSelector, filtering.end);
    const handle = useCallback(
        (value: Flag<EAviaSearchTransferFilter>) => {
            filtering.start(setTransferFilter(value));
            handleAsync(setTransferFilter, value);
        },
        [handleAsync, filtering],
    );
    const reset = useCallback(() => {
        filtering.start(resetFilterValue('transfer'));
        handleAsync(resetFilterValue, 'transfer');
    }, [handleAsync, filtering]);

    return {filter, handle, reset};
}
