import {parseDate, DateLikeType} from 'utilities/dateUtils';

type TransferTimes = {
    departureDate: DateLikeType;
    arrivalDate: DateLikeType;
};

// eslint-disable-next-line valid-jsdoc
/**
 * Проверяет, что ночная пересадка содержит интервал времени в районе от 0 до 6 часов
 *
 *       X (0ч)  Y (6ч)         X (0ч)  Y (6ч)
 * ------|~~~~~~~|--------------|~~~~~~~|-------
 * ----|==========|-----------------------------
 *     Arrival    Departure
 */
export function isNighttimeTransfer({
    departureDate,
    arrivalDate,
}: TransferTimes) {
    if (typeof departureDate === 'string' && typeof arrivalDate === 'string') {
        return isNighttimeTransferFast({departureDate, arrivalDate});
    }

    const departureDateTime = parseDate(departureDate);
    const arrivalDateTime = parseDate(arrivalDate);
    const dayChangedCount = departureDateTime
        .clone()
        .startOf('day')
        .diff(arrivalDateTime.clone().startOf('day'), 'days');

    // Если вылет в другой день, то точно была ночная пересадка
    if (dayChangedCount > 0) {
        return true;
    }

    // Проверяем, что интервал [0-6)ч соприкосается с интервалом Время прилёта - время вылета
    if (arrivalDateTime.hours() < 6 && departureDateTime.hours() > 0) {
        return true;
    }

    return false;
}

// eslint-disable-next-line valid-jsdoc
/**
 * Оптимизированная версия isNighttimeTransfer, которая работает
 * со строковыми датами в ISO-подобных форматах
 */
function isNighttimeTransferFast({
    departureDate,
    arrivalDate,
}: {
    departureDate: string;
    arrivalDate: string;
}) {
    const departureDateStr = departureDate.slice(0, 10);
    const departureHours = +departureDate.slice(11, 13);
    const arrivalDateStr = arrivalDate.slice(0, 10);
    const arrivalHours = +arrivalDate.slice(11, 13);

    if (departureDateStr !== arrivalDateStr) {
        return true;
    }

    if (arrivalHours < 6 && departureHours > 0) {
        return true;
    }

    return false;
}
