import {parse} from 'query-string';

import {EAviaActionLogActionName} from 'server/loggers/avia/AviaActionLog/types/EAviaActionLogActionName';
import {IShowLogVariant} from 'server/loggers/avia/AviaShowLog/types/IShowLogVariant';
import {
    IAviaSearchActionLogParamsClient,
    IAviaOrderActionLogParamsClient,
} from 'server/loggers/avia/AviaActionLog/types/IAviaActionLogParamsClient';
import {IAviaActionLogParamsServer} from 'server/loggers/avia/AviaActionLog/types/IAviaActionLogParamsServer';
import EAviaDynamicActionLogActionName from 'server/loggers/avia/AviaActionLog/types/EAviaDynamicActionLogActionName';
import {EAviaLinkSource} from 'types/avia/url/EAviaLinkSource';

import {INormalizedTDReference} from 'reducers/avia/utils/ticketDaemon/normalizeTDReference';

import {IAviaVariantGroup} from 'selectors/avia/utils/denormalization/variantGroup';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';
import {IResultAviaFlight} from 'selectors/avia/utils/denormalization/flight';

import {logError} from 'utilities/logger/logError';
import {
    aviaSearchActionLogPrepareParams,
    aviaOrderActionLogPrepareParams,
} from 'server/loggers/avia/AviaActionLog/utils/aviaActionLogPrepareParams';
import {getActionId} from 'server/loggers/avia/AviaVariantsLog/utils/getActionId';
import {getSession} from 'server/loggers/avia/AviaVariantsLog/utils/getSession';
import {getAviaPartnersInfo} from 'projects/avia/lib/logging/getPartnersInfo';
import getValueFromEnum from 'utilities/getValueFromEnum';

import {aviaLoggingBrowserProvider} from 'serviceProvider/avia/logging/aviaLoggingBrowserProvider';

const SESSION_ID = getSession();

function getDirectionRoute({departure, arrival, number}: IResultAviaFlight) {
    const datetimeDeparture = (departure && departure.local.slice(0, -3)) || '';
    const datetimeArrival = (arrival && arrival.local.slice(0, -3)) || '';
    const datetime = [datetimeDeparture, datetimeArrival].join('_');

    return number + '.' + datetime;
}

function getVariantData(
    record: IAviaVariantGroup,
    variant: IResultAviaVariant,
    reference: INormalizedTDReference,
): Omit<IShowLogVariant, 'position' | 'interesting'> {
    const forward = variant.route[0];
    const backward = variant.route[1] || [];
    const forwardInStr = forward.map(getDirectionRoute).join(',');
    const backwardInStr = backward.map(getDirectionRoute).join(',');

    return {
        partner: variant.price.partner.code,
        fromId: variant.fromCode,
        toId: variant.toCode,
        status: variant.status,
        baggage: variant.baggage,
        forward: forwardInStr,
        backward: backwardInStr,
        tariff: variant.price.tariff,
        aircompany_price: variant.price.fromCompany,
        variantId: variant.variantId,
        partnersInfo: getAviaPartnersInfo(record.variants, reference),
    };
}

function getSearchVariantData(
    record: IAviaVariantGroup,
    position: number,
    interesting: boolean,
    reference: INormalizedTDReference,
): IShowLogVariant {
    return {
        ...getVariantData(
            record,
            record.cheapestVariantsByTariff.cheapest,
            reference,
        ),
        position,
        interesting,
    };
}

function getOrderVariantData(
    variant: IResultAviaVariant,
    position: number,
    reference: INormalizedTDReference,
): IShowLogVariant {
    const forward = variant.route[0];
    const backward = variant.route[1] || [];
    const forwardInStr = forward.map(getDirectionRoute).join(',');
    const backwardInStr = backward.map(getDirectionRoute).join(',');

    return {
        partner: variant.price.partner.code,
        fromId: variant.fromCode,
        toId: variant.toCode,
        status: variant.status,
        baggage: variant.baggage,
        position,
        forward: forwardInStr,
        backward: backwardInStr,
        tariff: variant.price.tariff,
        aircompany_price: variant.price.fromCompany,
        variantId: variant.variantId,
        partnersInfo: getAviaPartnersInfo([variant], reference),
    };
}

function logShow(actionId: string, variants: IShowLogVariant[]): void {
    aviaLoggingBrowserProvider.show(variants, SESSION_ID, actionId).catch(e => {
        logError(
            {
                message:
                    '[YATRAVEL][AVIA] Ошибка логгирования показов вариантов',
                block: 'aviaVariantsLogger',
            },
            e,
        );
    });
}

export function logSearchOfferShow(
    actionId: string,
    record: IAviaVariantGroup,
    variant: IResultAviaVariant,
    reference: INormalizedTDReference,
    position: number,
    interesting: boolean,
) {
    logShow(actionId, [
        {
            ...getVariantData(record, variant, reference),
            position,
            interesting,
        },
    ]);
}

export function logSearchShow(
    actionId: string,
    records: IAviaVariantGroup[],
    start: number,
    interesting: boolean,
    reference: INormalizedTDReference,
): void {
    const variants = records.map((record, position) =>
        getSearchVariantData(record, start + position, interesting, reference),
    );

    logShow(actionId, variants);
}

export function logOrderShow(
    actionId: string,
    variants: IResultAviaVariant[],
    reference: INormalizedTDReference,
): void {
    const variantsData = variants.map((record, position) =>
        getOrderVariantData(record, position + 1, reference),
    );

    logShow(actionId, variantsData);
}

export function logAction(
    action: EAviaActionLogActionName | EAviaDynamicActionLogActionName,
    params: IAviaActionLogParamsServer,
): string {
    const actionName = params.page + action;
    const actionId = getActionId(actionName);
    const linkSource =
        params.linkSource ||
        getValueFromEnum(
            String(parse(window.location.search).linkSource),
            EAviaLinkSource,
        );

    aviaLoggingBrowserProvider
        .action(SESSION_ID, actionId, actionName, {
            ...params,
            linkSource,
        })
        .catch(e => {
            logError(
                {
                    message: '[YATRAVEL][AVIA] Ошибка логгирования действий',
                    block: 'aviaVariantsLogger',
                },
                e,
            );
        });

    return actionId;
}

export function logSearchAction(
    action: EAviaActionLogActionName,
    params: IAviaSearchActionLogParamsClient,
): string {
    return logAction(action, aviaSearchActionLogPrepareParams(params));
}

export function logOrderAction(
    action: EAviaActionLogActionName,
    params: IAviaOrderActionLogParamsClient,
): string {
    return logAction(action, aviaOrderActionLogPrepareParams(params));
}
