import {EFareFamilyTermCode} from 'server/api/AviaTicketDaemonApi/types/IAviaTDFareFamily';

import {INormalizedTDReference} from 'reducers/avia/utils/ticketDaemon/normalizeTDReference';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {isBaggageIncluded} from 'projects/avia/lib/baggage';
import IPrice from 'utilities/currency/PriceInterface';
import {TermsHelper} from 'projects/avia/pages/AviaOrder/components/FlightInfoFareFamily/utils/TermsHelper';

export interface IVariantBaggageAndTariffsInfo {
    price: IPrice;
    fareCodes: string[][];
    baggageAndCarryOn: string[][];
    fareFamilies: (string | null)[][];
}

export type TVariantPartnersInfo = Record<
    string,
    IVariantBaggageAndTariffsInfo
>;

function getBaggageAndCarryOn(variant: IResultAviaVariant): string[][] {
    const {route, price} = variant;

    return route.map((flight, flightIndex) =>
        flight.map((segment, segmentIndex) => {
            const baggage = price.partnersBaggage[flightIndex][segmentIndex];
            const baggageCount = isBaggageIncluded(baggage)
                ? baggage.pc?.count || 0
                : 0;
            const baggageWeight = baggageCount ? baggage.wt?.count || 0 : 0;
            const carryOnCount = segment.companyTariff?.carryon ? 1 : 0;
            const carryOnWeight = carryOnCount
                ? segment.companyTariff?.carryonNorm
                : 0;

            return [
                `${baggageCount}p${baggageWeight}kg`,
                `${carryOnCount}p${carryOnWeight}kg`,
            ].join('');
        }),
    );
}

function getFareFamilies(
    variant: IResultAviaVariant,
    reference: INormalizedTDReference,
): (string | null)[][] {
    return variant.price.fareFamilies.map(flightFamily =>
        flightFamily.map(segmentFamily => {
            if (!segmentFamily) {
                return null;
            }

            const family = reference.fareFamilies[segmentFamily];
            const {terms} = family;
            const baggageRule = terms[EFareFamilyTermCode.baggage];
            const carryOnRule = terms[EFareFamilyTermCode.carryOn];
            const baggageCount = baggageRule?.places || 0;
            const baggageWeight = baggageCount ? baggageRule?.weight || 0 : 0;
            const carryOnCount = carryOnRule?.places || 0;
            const carryOnWeight = carryOnCount ? carryOnRule?.weight || 0 : 0;

            return [
                [
                    `${baggageCount}p${baggageWeight}kg`,
                    `${carryOnCount}p${carryOnWeight}kg`,
                ].join(''),
                family.tariffGroupName,
                [
                    EFareFamilyTermCode.refundable,
                    TermsHelper.getAvailability(
                        terms[EFareFamilyTermCode.refundable],
                    ),
                ].join(':'),
                [
                    EFareFamilyTermCode.changingCarriage,
                    TermsHelper.getAvailability(
                        terms[EFareFamilyTermCode.changingCarriage],
                    ),
                ].join(':'),
                [
                    EFareFamilyTermCode.seatSelectionCheckIn,
                    TermsHelper.getAvailability(
                        terms[EFareFamilyTermCode.seatSelectionCheckIn],
                    ),
                ].join(':'),
                [
                    EFareFamilyTermCode.miles,
                    TermsHelper.getMiles(terms[EFareFamilyTermCode.miles]) || 0,
                ].join(':'),
            ].join(';');
        }),
    );
}

export function getAviaBaggageAndTariffsInfo(
    variant: IResultAviaVariant,
    reference: INormalizedTDReference,
): IVariantBaggageAndTariffsInfo {
    const {price} = variant;
    const {fareCodes, tariff} = price;

    return {
        fareCodes: fareCodes[0],
        price: tariff,
        baggageAndCarryOn: getBaggageAndCarryOn(variant),
        fareFamilies: getFareFamilies(variant, reference),
    };
}

export function getAviaPartnersInfo(
    variants: IResultAviaVariant[],
    reference: INormalizedTDReference,
): TVariantPartnersInfo {
    return variants.reduce((data, variant) => {
        const {price} = variant;
        const partnerCode = price.partner.code;

        // Логируем минимальную цену от партнёра
        if (
            data[partnerCode] &&
            data[partnerCode].price.value < price.tariff.value
        ) {
            return data;
        }

        return {
            ...data,
            [partnerCode]: getAviaBaggageAndTariffsInfo(variant, reference),
        };
    }, {} as TVariantPartnersInfo);
}
