import {DEFAULT_AVIA_CLASS} from 'constants/avia';

import {
    EOneWay,
    IAviaParams,
} from 'server/services/AviaSearchService/types/IAviaParams';
import {EAviaClassType} from 'types/avia/EAviaClassType';

/* eslint-disable camelcase */
export interface IDecodedQueryKey {
    fromId: string;
    toId: string;
    when: string;
    return_date?: string;
    klass?: string;
    adult_seats: number;
    children_seats: number;
    infant_seats: number;
    national_version: string;
    queryKey: string;
    [key: string]: unknown;
}

export interface IDecodedQid extends IDecodedQueryKey {
    lang: string;
}

function emptyQueryKey(): IDecodedQueryKey {
    return {
        fromId: '',
        toId: '',
        when: '',
        return_date: '',
        klass: '',
        adult_seats: 1,
        children_seats: 0,
        infant_seats: 0,
        national_version: '',
        queryKey: '',
    };
}

function emptyQid(): IDecodedQid {
    return {
        ...emptyQueryKey(),
        lang: '',
    };
}

export function decodeQueryKey(queryKey: string): IDecodedQueryKey {
    try {
        const [
            fromId,
            toId,
            when,
            return_date,
            klass,
            adultSeats,
            childrenSeats,
            infantSeats,
            national_version,
        ] = queryKey.split('_').map(v => (v === 'None' ? '' : v));

        return {
            fromId,
            toId,
            when,
            return_date,
            klass,
            national_version,
            queryKey,
            adult_seats: Number(adultSeats),
            children_seats: Number(childrenSeats),
            infant_seats: Number(infantSeats),
        };
    } catch (e) {
        return emptyQueryKey();
    }
}

export function decodeQidOrDefault(qid: Nullable<string>): IDecodedQid {
    if (!qid) {
        return emptyQid();
    }

    try {
        const [, , , queryKey, lang] = qid.split('.');

        return {
            ...decodeQueryKey(queryKey),
            lang,
        };
    } catch (e) {
        return emptyQid();
    }
}

export function qidToSearchForm(qid: string): IAviaParams {
    const {
        fromId,
        toId,
        when,
        adult_seats,
        children_seats,
        infant_seats,
        return_date,
        klass,
    } = decodeQidOrDefault(qid);

    return {
        fromId,
        toId,
        when,
        adult_seats: String(adult_seats),
        children_seats: String(children_seats),
        infant_seats: String(infant_seats),
        oneway: return_date ? EOneWay.ROUND_TRIP : EOneWay.ONE_WAY,
        klass:
            Object.values(EAviaClassType).find(k => k === klass) ??
            DEFAULT_AVIA_CLASS,
        return_date: return_date ?? '',
    };
}
