import {IRating} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';

import {CHAR_EM_DASH, CHAR_NBSP} from 'utilities/strings/charCodes';

import * as i18nBlock from 'i18n/avia-flightDelay';

type TStatusWeightKey =
    | 'delayedLess30'
    | 'delayed3060'
    | 'delayed6090'
    | 'delayedMore90'
    | 'canceled';

type TStatusWeight = Record<TStatusWeightKey, number>;

const statusWeight: TStatusWeight = {
    delayedLess30: 0,
    delayed3060: 1,
    delayed6090: 2,
    delayedMore90: 3,
    canceled: 4,
};

type TPrefixName = 'often' | 'rarely' | 'occasionally';

type TPrefix = {name: TPrefixName; value: number};

const prefixes: TPrefix[] = [
    {name: 'often', value: 16},
    {name: 'rarely', value: 7},
    {name: 'occasionally', value: 3},
];

function getPrefixName(value: number): TPrefixName | null {
    const prefix = prefixes.find(_prefix => value >= _prefix.value);

    return prefix?.name ?? null;
}

function getStatusSorter(rating: IRating) {
    return function (a: TStatusWeightKey, b: TStatusWeightKey): number {
        if (!rating[a]) {
            return 1;
        }

        if (!rating[b]) {
            return -1;
        }

        if (rating[a] === rating[b]) {
            return Number(statusWeight[b]) - Number(statusWeight[a]);
        }

        return Number(rating[b]) - Number(rating[a]);
    };
}

function getWorstStatus(rating: IRating): string | null {
    if (!rating) {
        return null;
    }

    return (Object.keys(statusWeight) as TStatusWeightKey[]).sort(
        getStatusSorter(rating),
    )[0];
}

export function hasInfo(rating: IRating & PartialRecord<string, any>): boolean {
    return rating && rating[String(getWorstStatus(rating))] >= 3;
}

type TInfo = {
    shortText: string;
    intime?: boolean;
    worst?: string;
};

export function getInfo(rating: IRating & PartialRecord<string, any>): TInfo {
    const worstStatus = getWorstStatus(rating);
    const prefixName = getPrefixName(rating[String(worstStatus)]);

    if (!worstStatus || !rating[worstStatus] || !prefixName) {
        return {
            shortText: i18nBlock.intime().toLowerCase(),
            intime: true,
        };
    }

    return {
        shortText: `${getFlightDelayText(
            worstStatus,
        )}${CHAR_NBSP}${CHAR_EM_DASH} ${rating[worstStatus]}%`.toLowerCase(),
        worst: worstStatus,
    };
}

function getFlightDelayText(worstStatus: string): string {
    switch (worstStatus) {
        case 'canceled':
            return i18nBlock.canceled();
        case 'default':
            return i18nBlock._default();
        case 'delayed':
            return i18nBlock.delayed();
        case 'delayed3060':
            return i18nBlock.delayed3060();
        case 'delayed6090':
            return i18nBlock.delayed6090();
        case 'delayedLess30':
            return i18nBlock.delayedLess30();
        case 'delayedMore90':
            return i18nBlock.delayedMore90();
        case 'intime':
            return i18nBlock.intime();
        case 'occasionally':
            return i18nBlock.occasionally();
        case 'often':
            return i18nBlock.often();
        case 'rarely':
            return i18nBlock.rarely();
        default:
            return '';
    }
}
