import Cookie from 'js-cookie';

import {ICookies} from '@yandex-data-ui/core/lib/types';

import {IAviaSearchResultsSorting} from 'reducers/avia/search/results/reducer';
import {EAviaResultsSortType} from 'reducers/avia/search/results/EAviaResultsSortType';

import {ESortDirection} from 'projects/avia/lib/comparators/TComparator';

const AVIA_SORT_COOKIE_KEY = 'ya-travel-avia-search-sort';

export function setSortingCookie(sorting: IAviaSearchResultsSorting) {
    const str = sortingToStr(sorting);

    const expiresDate = new Date();
    expiresDate.setDate(expiresDate.getDate() + 10);

    Cookie.set(AVIA_SORT_COOKIE_KEY, str, {
        expires: expiresDate,
    });
}

export function getSortingFromCookies(cookies: ICookies) {
    return strToSorting(cookies[AVIA_SORT_COOKIE_KEY]);
}

export function getSortingFromCookie(): IAviaSearchResultsSorting | null {
    const str = Cookie.get(AVIA_SORT_COOKIE_KEY);
    return strToSorting(str);
}

const SORT_TYPE_TO_STR: Record<EAviaResultsSortType, string> = {
    [EAviaResultsSortType.ARRIVAL]: 'a',
    [EAviaResultsSortType.DEPARTURE]: 'd',
    [EAviaResultsSortType.DURATION]: 'u',
    [EAviaResultsSortType.PRICE]: 'p',
    [EAviaResultsSortType.INTEREST]: '',
};

const STR_TO_SORT_TYPE: Record<string, EAviaResultsSortType> = {
    [SORT_TYPE_TO_STR[EAviaResultsSortType.ARRIVAL]]:
        EAviaResultsSortType.ARRIVAL,
    [SORT_TYPE_TO_STR[EAviaResultsSortType.DEPARTURE]]:
        EAviaResultsSortType.DEPARTURE,
    [SORT_TYPE_TO_STR[EAviaResultsSortType.DURATION]]:
        EAviaResultsSortType.DURATION,
    [SORT_TYPE_TO_STR[EAviaResultsSortType.PRICE]]: EAviaResultsSortType.PRICE,
};

const SORT_DIR_TO_STR: Record<ESortDirection, string> = {
    [ESortDirection.ASCENDING]: 'a',
    [ESortDirection.DESCENDING]: 'd',
};

const STR_TO_SORT_DIR: Record<string, ESortDirection> = {
    [SORT_DIR_TO_STR[ESortDirection.ASCENDING]]: ESortDirection.ASCENDING,
    [SORT_DIR_TO_STR[ESortDirection.DESCENDING]]: ESortDirection.DESCENDING,
};

function sortingToStr({type, direction}: IAviaSearchResultsSorting) {
    return `${SORT_TYPE_TO_STR[type]}:${SORT_DIR_TO_STR[direction]}`;
}

function strToSorting(
    str: string | null | undefined,
): IAviaSearchResultsSorting | null {
    if (!str) {
        return null;
    }

    const [typeStr, directionStr] = str.split(':');
    const type = STR_TO_SORT_TYPE[typeStr];
    const direction = STR_TO_SORT_DIR[directionStr];

    if (!type || !direction) {
        return null;
    }

    return {
        direction,
        type,
    };
}
