import {parse, stringify} from 'query-string';
import {reduce} from 'lodash';

import {
    IAviaSearchResultsFilters,
    TAviaFilterKey,
} from 'reducers/avia/search/results/filters/reducer';

import {airportsFilterConverter} from './airportsFilterConverter';
import {baggageFilterConverter} from './baggageFilterConverter';
import {priceFilterConverter} from './priceFilterConverter';
import {companyFilterConverter} from './companyFilterConverter';
import {partnersFilterConverter} from './partnersFilterConverter';
import {timeFilterConverter} from './timeFilterConverter';
import {transferFilterConverter} from './transferFilterConverter';
import {plusPointsFilterConverter} from './plusPointsFilterConverter';

const SHORT_KEY_TO_FILTER_KEY: Record<string, TAviaFilterKey> = {
    a: 'airports',
    bg: 'baggage',
    c: 'company',
    pt: 'partners',
    budget: 'price',
    t: 'time',
    tt: 'transfer',
    pp: 'plusPoints',
};

const FILTER_KEY_TO_SHORT_KEY: Record<TAviaFilterKey, string> = {
    airports: 'a',
    baggage: 'bg',
    company: 'c',
    partners: 'pt',
    price: 'budget',
    time: 't',
    transfer: 'tt',
    plusPoints: 'pp',
};

function isFilterKey(key: unknown): key is TAviaFilterKey {
    return typeof key === 'string' && key in FILTER_KEY_TO_SHORT_KEY;
}

export type TFilterConverter<T> = {
    fromString(str: string): T;
    toString(value: T): string;
};

type TFilterConverters = {
    [Key in TAviaFilterKey]: TFilterConverter<IAviaSearchResultsFilters[Key]>;
};

const CONVERTERS: TFilterConverters = {
    airports: airportsFilterConverter,
    baggage: baggageFilterConverter,
    company: companyFilterConverter,
    partners: partnersFilterConverter,
    price: priceFilterConverter,
    time: timeFilterConverter,
    transfer: transferFilterConverter,
    plusPoints: plusPointsFilterConverter,
};

export type TPartialFilterValues = Partial<IAviaSearchResultsFilters>;

export function hashToFilterValues(hash: string): TPartialFilterValues {
    const valuesMap = parse(hash, {}) as Record<string, string>;

    return reduce(
        valuesMap,
        (filters, strValue, shortKey) => {
            const filterKey = SHORT_KEY_TO_FILTER_KEY[shortKey];

            if (!filterKey) {
                return filters;
            }

            const {fromString} = CONVERTERS[filterKey];

            filters[filterKey] = fromString(strValue) as any;

            return filters;
        },
        {} as TPartialFilterValues,
    );
}

export const EMPTY_FILTER_VALUES = 'empty';

export function filterValuesToHash(values: TPartialFilterValues): string {
    const valuesMap = reduce(
        values,
        (acc, value, filterKey) => {
            if (!isFilterKey(filterKey)) {
                return acc;
            }

            const shortKey = FILTER_KEY_TO_SHORT_KEY[filterKey];
            const {toString} = CONVERTERS[filterKey];

            acc[shortKey] = toString(value as any);

            return acc;
        },
        {} as Record<string, string>,
    );

    if (Object.keys(valuesMap).length === 0) {
        return EMPTY_FILTER_VALUES;
    }

    return stringify(valuesMap, {encode: false});
}
