import {
    EAviaSearchTimeFilter,
    IAviaSearchTimeFilter,
} from 'reducers/avia/search/results/filters/reducer';

import * as Flags from 'utilities/flags/flags';

import {TFilterConverter} from './index';

const TIMES = [
    EAviaSearchTimeFilter.MORNING,
    EAviaSearchTimeFilter.DAY,
    EAviaSearchTimeFilter.EVENING,
    EAviaSearchTimeFilter.NIGHT,
];

export const timeFilterConverter: TFilterConverter<IAviaSearchTimeFilter[]> = {
    fromString(str) {
        const split = str.split('b');
        if (split.length < 2) {
            return [];
        }
        const forward = split[0].slice(1);
        const backward = split[1];
        const forwardParts = forward.split('-');
        const backwardParts = backward.split('-');

        return [
            getDirectionValue(forwardParts),
            getDirectionValue(backwardParts),
        ];
    },
    toString(value) {
        if (value.length === 0) {
            return '';
        }
        const [forward, backward] = value;
        return `f|${timesToString(forward)}b|${timesToString(backward)}`;
    },
};

function getDirectionValue(directions: string[]): IAviaSearchTimeFilter {
    return directions.reduce(
        (result, directionStr) => {
            const direction = directionStr.split('|');
            const type = direction[1] === 'a' ? 'arrival' : 'departure';
            const times = (direction[2] || '')
                .split(',')
                .filter(Boolean)
                .map(time => TIMES[Number(time)])
                .reduce(
                    Flags.enable,
                    Flags.createFlags<EAviaSearchTimeFilter>(0),
                );
            result[type] = times;
            return result;
        },
        {
            arrival: 0 as Flags.Flag<EAviaSearchTimeFilter>,
            departure: 0 as Flags.Flag<EAviaSearchTimeFilter>,
        },
    );
}

function timeToString(time: Flags.Flag<EAviaSearchTimeFilter>) {
    return Flags.toArray(time)
        .map(flag => TIMES.indexOf(flag))
        .filter(x => x !== -1)
        .join(',');
}
function timesToString(times: IAviaSearchTimeFilter | undefined) {
    if (!times) {
        return '';
    }
    const parts = [];
    if (times.departure) {
        parts.push(`d|${timeToString(times.departure)}`);
    }

    if (times.arrival) {
        parts.push(`a|${timeToString(times.arrival)}`);
    }

    return parts.join('-|');
}
