import {TAviaSearchCompanyFilter} from 'reducers/avia/search/results/filters/reducer';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';
import {IResultAviaFlight} from 'selectors/avia/utils/denormalization/flight';
import {IResultCompany} from 'selectors/avia/utils/denormalization/company';

/**
 * Если у варианта оперирующая авиакомпания отличается от основной, то мы хотим фильровать по ней
 * Например, рейс Аэрофлота - оперируемый Победой - показывается при фильтрации Победы
 * и скрывается при фильтрации Аэрофлота.
 */
export function getFilterCompanyFromFlight(
    flight: IResultAviaFlight,
): IResultCompany {
    return flight.operating && flight.operating.company.id !== flight.company.id
        ? flight.operating.company
        : flight.company;
}

export function getFilterCompaniesCountFromVariant(
    variant: IResultAviaVariant,
): number {
    const [forward, backward] = variant.route;
    const flights = forward.concat(backward);

    const companiesIds = new Set();

    flights.forEach(flight => {
        const company = getFilterCompanyFromFlight(flight);

        companiesIds.add(company.id);
    });

    return companiesIds.size;
}

/**
 * - Узнаем авикомпанию варианта из первого сниппета перелета туда.
 * - В фильтр выводятся только АК, у которых есть чисто их варианты
 * - При этом пересадки при выборе нескольких АК тоже фильтруются, но не влияют на цену и галочки в фильтре.
 * - Если в варианте больше одной авиакомпании,
 * вариант будет помечен, как неподходящий при инициализации фильтра.
 */
export function getCompanyFromVariant(
    variant: IResultAviaVariant,
): IResultCompany {
    return getFilterCompanyFromFlight(variant.forward[0]);
}

export function isPassingCompanyFilter(
    variant: IResultAviaVariant,
    companyFilter: TAviaSearchCompanyFilter,
): boolean {
    if (companyFilter.companiesIds.length) {
        const companiesIds = new Set(companyFilter.companiesIds);
        const [forward, backward] = variant.route;
        const flights = forward.concat(backward);
        const comparator = (flight: IResultAviaFlight) => {
            const company = getFilterCompanyFromFlight(flight);

            return companiesIds.has(company.id);
        };

        if (companyFilter.combinationsAreEnabled) {
            return flights.some(comparator);
        }

        return flights.every(comparator);
    }

    return true;
}
