import {
    AVIA_CLASSES,
    MAX_PASSENGERS_COUNT,
    MIN_ADULTS_COUNT,
    MIN_CHILDREN_COUNT,
    MIN_INFANTS_COUNT,
} from 'constants/avia';

import {
    EOneWay,
    IAviaParams,
} from 'server/services/AviaSearchService/types/IAviaParams';

import {isValidDate} from 'projects/avia/lib/date/isValidDate';
import {parseAviaPeriod} from 'projects/avia/lib/search/parsePeriod';
import {isValidActualDate} from 'projects/avia/lib/date/isValidActualDate';

function getSeatsCount(value?: string): number {
    const num = Number(value);

    return isNaN(num) ? 0 : num;
}

function isValidPassengersCount(num: number, min: number): boolean {
    return !isNaN(num) && num >= min && num <= MAX_PASSENGERS_COUNT;
}

// eslint-disable-next-line complexity
export function sanitizeAviaSearchParams(
    params: Partial<IAviaParams>,
    outdateIsValid: boolean = false,
): Partial<IAviaParams> {
    const result: Partial<IAviaParams> = {};
    const {fromName, toName, fromId, toId, when} = params;
    const parsedWhen = typeof when === 'string' ? parseAviaPeriod(when) : when;
    const dateChecker = outdateIsValid ? isValidDate : isValidActualDate;

    if (fromName) {
        result.fromName = fromName;
    }

    if (toName) {
        result.toName = toName;
    }

    if (fromId) {
        result.fromId = fromId;
    }

    if (toId) {
        result.toId = toId;
    }

    if (dateChecker(parsedWhen)) {
        result.when = parsedWhen;
    }

    // Пустое значение для даты возврата - это ок
    if (params.return_date === '') {
        result.return_date = params.return_date;
    } else if (params.return_date && dateChecker(params.return_date)) {
        result.return_date = params.return_date;
    }

    const adultSeats = getSeatsCount(params.adult_seats);
    const childrenSeats = getSeatsCount(params.children_seats);
    const infantSeats = getSeatsCount(params.infant_seats);
    const seatsCount = adultSeats + childrenSeats + infantSeats;

    if (
        isValidPassengersCount(
            seatsCount,
            MIN_ADULTS_COUNT + MIN_CHILDREN_COUNT + MIN_INFANTS_COUNT,
        )
    ) {
        if (
            params.adult_seats &&
            isValidPassengersCount(adultSeats, MIN_ADULTS_COUNT)
        ) {
            result.adult_seats = params.adult_seats;
        }

        if (
            params.children_seats &&
            isValidPassengersCount(childrenSeats, MIN_CHILDREN_COUNT)
        ) {
            result.children_seats = params.children_seats;
        }

        if (
            params.infant_seats &&
            isValidPassengersCount(infantSeats, MIN_INFANTS_COUNT)
        ) {
            result.infant_seats = params.infant_seats;
        }
    }

    if (params.klass && AVIA_CLASSES.includes(params.klass)) {
        result.klass = params.klass;
    }

    if (
        params.oneway === EOneWay.ONE_WAY ||
        params.oneway === EOneWay.ROUND_TRIP
    ) {
        result.oneway = params.oneway;
    }

    if (params.baggage === '1') {
        result.baggage = '1';
    }

    if (params.free_refund === '1') {
        result.free_refund = '1';
    }

    if (params.charge_refund === '1') {
        result.charge_refund = '1';
    }

    return result;
}
