import {URLs} from 'constants/urls';

import {EAviaClassType} from 'types/avia/EAviaClassType';
import {IPassengers} from 'projects/avia/pages/AviaBooking/types/IPassengers';
import {TSettlementKey, TStationKey} from 'types/PointKey';
import TDateRobot from 'types/common/date/TDateRobot';
import {EAviaLinkSource} from 'types/avia/url/EAviaLinkSource';
import {IUtmQueryParams} from 'types/common/IQueryParams';
import {EOneWay} from 'server/services/AviaSearchService/types/IAviaParams';

import {IAviaSearchResultsFilters} from 'reducers/avia/search/results/filters/reducer';

import {internalUrl} from 'utilities/url';
import getFiltersForUrl from 'projects/avia/utilities/getFiltersForUrl';

interface IGetSearchUrl {
    fromKey: TSettlementKey | TStationKey;
    toKey: TSettlementKey | TStationKey;
    dateForward: TDateRobot;

    dateBackward?: TDateRobot;
    klass?: EAviaClassType;
    passengers?: IPassengers;
    filters?: Partial<IAviaSearchResultsFilters>;
    /** Название пункта "откуда" */
    fromName?: string;
    /** Название пункта "куда" */
    toName?: string;
    /**
     * Источник ссылки. Пример: 'searchToCountry'.
     * Этот параметр логируется в action-log поиска.
     * Применяется для отсеживания конверсии с разных источников трафика.
     */
    linkSource?: EAviaLinkSource;
    utm?: IUtmQueryParams;
}

/**
 * Возвращает параметр "oneway" для урла поиска авиа
 * 1 - только туда
 * 2 - туда и обратно
 */
export function getOneWayParam(dateBackward: TDateRobot | undefined): EOneWay {
    return dateBackward ? EOneWay.ROUND_TRIP : EOneWay.ONE_WAY;
}

/**
 * Возвращает урл страницы поиска для заданных параметров.
 *
 * Появилась как типизированная замена aviaURLs.getSearchResultsUrl
 */
export function getSearchUrl({
    fromKey,
    toKey,
    dateForward,
    dateBackward,
    klass,
    passengers,
    filters,
    fromName,
    toName,
    linkSource,
    utm,
}: IGetSearchUrl): string {
    const query = {
        ...utm,
        adult_seats: passengers?.adults,
        children_seats: passengers?.children,
        infant_seats: passengers?.infants,
        fromId: fromKey,
        toId: toKey,
        when: dateForward,
        return_date: dateBackward,
        oneway: getOneWayParam(dateBackward),
        klass,
        fromName,
        toName,
        linkSource,
    };

    const fragment = filters && getFiltersForUrl(filters);

    return internalUrl(URLs.aviaSearchResults, query, null, fragment);
}
