import {generatePath} from 'react-router-dom';

import {URLs} from 'constants/urls';

import {
    IBookingFlightsByRoute,
    IBookingRouteFlight,
} from 'server/services/AviaBookingService/variants/types';
import {IBookingDateTime} from 'server/api/AviaBookingApi/types/IBookingDateTime';
import {IInternalUrlOptions, IUrlQuery} from 'utilities/url/types';
import {EAviaLinkSource} from 'types/avia/url/EAviaLinkSource';
import {EOneWay} from 'server/services/AviaSearchService/types/IAviaParams';

import {IResultVariantPrice} from 'selectors/avia/utils/denormalization/prices';

import {internalUrl} from 'utilities/url';
import {isValidDate, parseDate} from 'utilities/dateUtils';
import {qidToSearchForm} from 'projects/avia/lib/qid';
import {restoreYtpReferer} from 'utilities/ytpReferer';

const BASE_AVIA_URL = URLs.avia;

export interface ICreateUrlToRedirectPageParams {
    forwardRoute: string;
    backwardRoute: string;
    price: IResultVariantPrice;
    hasPartnersBaggage: boolean;
    variantId: string;
    variantTestContext?: string;
    paymentTestContext?: string;
    qid: string;
    promoId?: string;
    ytpReferer?: string;
    boy?: boolean;
}

export const aviaURLs = {
    getBookingFormUrl(token: string, query?: Record<string, string>): string {
        return internalUrl(
            URLs.aviaBooking,
            {
                ...query,
                token,
            },
            {filterNull: true},
        );
    },
    getBookingResultUrl(
        orderId: string,
        query?: Record<string, string>,
    ): string {
        const url = URLs.aviaBookingResult.replace(':orderId', orderId);

        return internalUrl(url, query, {filterNull: true});
    },

    getBookingPaymentUrl(
        orderId: string,
        variantToken: string,
        query?: Record<string, string>,
    ): string {
        const url = URLs.aviaBookingPayment
            .replace(':token', variantToken)
            .replace(':orderId', orderId);

        return internalUrl(url, query, {filterNull: true});
    },

    /** Лучше использовать типизированную версию src/projects/avia/lib/urls/getSearchUrl.ts */
    getSearchResultsUrl(
        query?: (IUrlQuery & {linkSource?: EAviaLinkSource}) | null,
        fragment?: IUrlQuery,
    ): string {
        const validQuery = {
            ...query,
            oneway: EOneWay.ONE_WAY,
        };

        if (query && query.return_date && isValidDate(query.return_date)) {
            validQuery.oneway = EOneWay.ROUND_TRIP;
        }

        return internalUrl(URLs.aviaSearchResults, validQuery, null, fragment);
    },
    getSearchResultsUrlByQid(qid: string): string {
        return this.getSearchResultsUrl(qidToSearchForm(qid));
    },

    getOrderUrl(query?: IUrlQuery, fragment?: IUrlQuery): string {
        return internalUrl(URLs.aviaOrder, query, null, fragment);
    },
    getOrderUrlByQidAndRouteFlights(
        qid: string,
        routeFlights: IBookingFlightsByRoute,
    ): string {
        return this.getOrderUrl({
            ...qidToSearchForm(qid),
            forward: getFlightNumberDateTime(routeFlights.forward),
            backward: getFlightNumberDateTime(routeFlights.backward),
        });
    },

    getUrlToRedirectPage(query?: IUrlQuery): string {
        return internalUrl(URLs.aviaRedirect, query);
    },

    getUrlToRedirectErrorPage(query?: IUrlQuery): string {
        return internalUrl(URLs.aviaRedirectError, query);
    },

    getUrlToRedirectUrlApi(query?: IUrlQuery): string {
        return internalUrl(`/api${BASE_AVIA_URL}/redirect/redirect`, query);
    },

    createUrlToRedirectPage(params: ICreateUrlToRedirectPageParams): string {
        return aviaURLs.getUrlToRedirectPage({
            forward: params.forwardRoute,
            backward: params.backwardRoute,
            partner: params.price.partner.code,
            withBaggage: Number(params.hasPartnersBaggage),
            fareFamiliesHash: params.price.fareFamiliesHash,
            tariff_sign: params.price.tariffSign,
            start: Date.now().toString(),
            variantId: params.variantId,
            variantTestContext: params.variantTestContext,
            paymentTestContext: params.paymentTestContext,
            qid: params.qid,
            promoId: params.promoId,
            ytpReferer: params.ytpReferer || restoreYtpReferer(),
            boy: params.boy ? '1' : undefined,
        });
    },

    getRoutePageUrl(
        fromSlug: string,
        toSlug: string,
        query?: IUrlQuery,
        options?: IInternalUrlOptions,
    ): string {
        const path = generatePath(URLs.aviaRoute, {fromSlug, toSlug});

        return internalUrl(path, query, options);
    },

    getFlightsToPageUrl(
        toSlug: string,
        query?: IUrlQuery,
        options?: IInternalUrlOptions,
    ): string {
        const path = generatePath(URLs.aviaFlightsTo, {toSlug});

        return internalUrl(path, query, options);
    },

    getAviaRoot(query?: IUrlQuery): string {
        return internalUrl(BASE_AVIA_URL, query);
    },
};

function getFlightNumberDateTime(flights: IBookingRouteFlight[]): string {
    return flights
        .map(flight => `${flight.number}.${formatDateTime(flight.departure)}`)
        .join(',');
}

function formatDateTime(dateTime: IBookingDateTime): string {
    return parseDate(dateTime.local).format('YYYY-MM-DDTHH:mm');
}
