import moment from 'moment';

import {DateLikeType, parseDate} from 'utilities/dateUtils';
import {ROBOT} from 'utilities/dateUtils/formats';

export type Validator<T> = (value: T) => [boolean, string];

export interface ValidationResult {
    valid: boolean;
    message: string;
}

export const EMAIL_RE = /^([#$%&'*+-/=?^_`{}|\w]+)@([\w-]+)\.(\w{2,})$/;
export const NO_RUSSIAN_LETTERS = /^[^а-яёА-ЯЁ]*$/;

/**
 * Возвращает успешный результат валидации
 */
export function valid() {
    return {valid: true, message: ''};
}

/**
 * Базовая функция валидации значения. Принимает в себя
 * значение для валидации и список валидаторов. Валидаторы применяются
 * в прямом порядке. Возвращает первую ошибку
 *
 * @param {Value} value — элемент для валидации
 * @param {Array<Validator<Value>>} fns - список валидаторов
 * @returns {ValidationResult} - результат валидациц
 * @template Value
 */
export function validate<Value>(
    value: Value,
    fns: Validator<Value>[],
): ValidationResult {
    for (const validator of fns) {
        const [valid, message] = validator(value);
        if (!valid) {
            return {valid, message};
        }
    }
    return valid();
}

export function pipeValidators<Value>(
    ...args: Validator<Value>[]
): Validator<Value> {
    return input => {
        const result = validate(input, args);
        return [result.valid, result.message];
    };
}

function matches(re: RegExp, message: string): Validator<string> {
    return (value: string) => [re.test(value), message];
}

function minMaxLen(
    min: number,
    max: number,
    message: string,
): Validator<string> {
    return ({length}: string) => [length <= max && length >= min, message];
}

function minLen(min: number, message: string): Validator<string> {
    return minMaxLen(min, Infinity, message);
}

function maxLen(max: number, message: string): Validator<string> {
    return minMaxLen(0, max, message);
}

function date(message: string): Validator<string> {
    return value => {
        const formatIsValid = /^\d\d\d\d-\d\d-\d\d$/.test(value);
        const dateIsValid = parseDate(value, ROBOT).isValid();
        return [formatIsValid && dateIsValid, message];
    };
}

function minDate(minDate: DateLikeType, message: string): Validator<string> {
    return value => [parseDate(value, ROBOT).isAfter(minDate), message];
}

function ageByDate(
    minAge: number,
    maxAge: number,
    relativeDate: DateLikeType,
    message: string,
): Validator<string> {
    return (value: string) => {
        const date = parseDate(value, ROBOT);
        const agesByDate = moment(relativeDate).diff(date, 'years');

        return [agesByDate >= minAge && agesByDate < maxAge, message];
    };
}

function required(message: string): Validator<string> {
    return (value: string) => [Boolean(value), message];
}

/**
 * Набор строковых валидаторов
 */
export const str = {
    matches,
    minMaxLen,
    minLen,
    maxLen,
    minDate,
    date,
    ageByDate,
    required,
};
