import React, {useCallback, useMemo, useState} from 'react';

import {EFormGroup} from '../../../../types/IBookingFormData';
import {EAgeCategory} from 'projects/avia/pages/AviaBooking/types/AgeCategoryType';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {getAviaPassengerTypesCounter} from 'projects/avia/pages/AviaBooking/components/AviaBookingForm/utilities/getAviaPassengerTypesCounter';

import Form from 'components/Form/Form';
import Separator from 'components/Separator/Separator';
import Intersperse from 'components/Intersperse/Intersperse';
import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import BookingContactsForm from 'components/BookingContactsForm/BookingContactsForm';
import TextMessage from 'projects/avia/pages/AviaBooking/components/AviaBookingForm/components/TextMessage/TextMessage';

import {IAviaBookingFormProps} from '../../AviaBookingForm';
import {AviaBookingPassengerForm} from '../AviaBookingPassengerForm/AviaBookingPassengerForm';

import cx from './AviaBookingFormDesktop.scss';

interface IAviaBookingFormDesktopProps extends IAviaBookingFormProps {}

const AviaBookingFormDesktop: React.FC<IAviaBookingFormDesktopProps> =
    props => {
        const {
            countries,
            disabled,
            flightType,
            allowedLoyaltyPrograms,
            types,
            passengersGroupedByType,
        } = props;

        const typesCounter = getAviaPassengerTypesCounter();

        const [isTranslatedMessageVisible, setIsTranslatedMessageVisible] =
            useState<{[index: number]: boolean}>({});
        const showTranslatedMessage = useCallback(
            (passengerIndex: number, value: boolean = true) => {
                if (isTranslatedMessageVisible[passengerIndex] !== value) {
                    setIsTranslatedMessageVisible({
                        ...isTranslatedMessageVisible,
                        [passengerIndex]: value,
                    });
                }
            },
            [isTranslatedMessageVisible],
        );

        const passengerForms = useMemo(() => {
            return types.map((type: EAgeCategory, index: number) => {
                const formQaAttributes = prepareQaAttributes({
                    key: `${type}-${typesCounter(type)}`,
                    current: 'passengers',
                    parent: props,
                });

                return (
                    <AviaBookingPassengerForm
                        key={index}
                        type={type}
                        index={index}
                        flightType={flightType}
                        formQaAttributes={formQaAttributes}
                        disabled={disabled}
                        message={
                            <TextMessage
                                className={cx('message')}
                                isVisible={isTranslatedMessageVisible[index]}
                            />
                        }
                        passengers={passengersGroupedByType[type]}
                        countries={countries}
                        allowedLoyaltyPrograms={allowedLoyaltyPrograms}
                        showTranslatedMessage={showTranslatedMessage}
                    />
                );
            });
        }, [
            types,
            flightType,
            isTranslatedMessageVisible,
            passengersGroupedByType,
            countries,
            disabled,
            typesCounter,
            allowedLoyaltyPrograms,
            showTranslatedMessage,
            props,
        ]);

        return (
            <BookingLayout.Forms>
                <BookingLayout.Card sectioned>
                    <Intersperse separator={<Separator />}>
                        {passengerForms}
                    </Intersperse>
                </BookingLayout.Card>

                <BookingLayout.Card>
                    <Form.FieldGroup groupId={EFormGroup.contacts}>
                        <BookingContactsForm
                            disabled={disabled}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'contacts',
                            })}
                        />
                    </Form.FieldGroup>
                </BookingLayout.Card>
            </BookingLayout.Forms>
        );
    };

export default AviaBookingFormDesktop;
