import React, {useCallback, useEffect, useState, useMemo} from 'react';

import {EFormGroup} from 'projects/avia/pages/AviaBooking/types/IBookingFormData';
import {EAgeCategory} from 'projects/avia/pages/AviaBooking/types/AgeCategoryType';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {getAviaPassengerTypesCounter} from 'projects/avia/pages/AviaBooking/components/AviaBookingForm/utilities/getAviaPassengerTypesCounter';

import Form from 'components/Form/Form';
import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import BookingContactsForm from 'components/BookingContactsForm/BookingContactsForm';
import TextMessage from 'projects/avia/pages/AviaBooking/components/AviaBookingForm/components/TextMessage/TextMessage';
import Snackbar from 'components/Snackbar/Snackbar';

import {IAviaBookingFormProps} from '../../AviaBookingForm';
import {AviaBookingPassengerForm} from '../AviaBookingPassengerForm/AviaBookingPassengerForm';

import cx from './AviaBookingFormMobile.scss';

interface IAviaBookingFormDesktopProps extends IAviaBookingFormProps {}

enum ETranslatedMessageState {
    HAS_NOT_SHOWN,
    SHOWING,
    HAS_BEEN_SHOWN,
}

const TRANSLATED_MESSAGE_TIMEOUT = 10000;

const AviaBookingFormMobile: React.FC<IAviaBookingFormDesktopProps> = props => {
    const {
        countries,
        disabled,
        flightType,
        allowedLoyaltyPrograms,
        types,
        passengersGroupedByType,
    } = props;

    const typesCounter = getAviaPassengerTypesCounter();

    const [isTranslatedMessageVisible, setIsTranslatedMessageVisible] =
        useState<ETranslatedMessageState>(
            ETranslatedMessageState.HAS_NOT_SHOWN,
        );

    const hideTranslatedMessage = useCallback(
        () =>
            setIsTranslatedMessageVisible(
                ETranslatedMessageState.HAS_BEEN_SHOWN,
            ),
        [],
    );

    useEffect(() => {
        if (isTranslatedMessageVisible === ETranslatedMessageState.SHOWING) {
            setTimeout(
                () => hideTranslatedMessage(),
                TRANSLATED_MESSAGE_TIMEOUT,
            );
        }
    }, [isTranslatedMessageVisible, hideTranslatedMessage]);

    const showTranslatedMessage = useCallback(() => {
        if (
            isTranslatedMessageVisible !==
            ETranslatedMessageState.HAS_BEEN_SHOWN
        ) {
            setIsTranslatedMessageVisible(ETranslatedMessageState.SHOWING);
        }
    }, [isTranslatedMessageVisible]);

    const passengerForms = useMemo(() => {
        return types.map((type: EAgeCategory, index: number) => {
            const formQaAttributes = prepareQaAttributes({
                key: `${type}-${typesCounter(type)}`,
                current: 'passengers',
                parent: props,
            });

            return (
                <AviaBookingPassengerForm
                    key={index}
                    type={type}
                    index={index}
                    flightType={flightType}
                    formQaAttributes={formQaAttributes}
                    disabled={disabled}
                    passengers={passengersGroupedByType[type]}
                    countries={countries}
                    allowedLoyaltyPrograms={allowedLoyaltyPrograms}
                    showTranslatedMessage={showTranslatedMessage}
                />
            );
        });
    }, [
        allowedLoyaltyPrograms,
        countries,
        disabled,
        flightType,
        passengersGroupedByType,
        showTranslatedMessage,
        types,
        typesCounter,
        props,
    ]);

    return (
        <>
            <BookingLayout.Forms>
                {passengerForms}
                <BookingLayout.Card>
                    <Form.FieldGroup groupId={EFormGroup.contacts}>
                        <BookingContactsForm
                            disabled={disabled}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'contacts',
                            })}
                        />
                    </Form.FieldGroup>
                </BookingLayout.Card>
            </BookingLayout.Forms>

            <Snackbar
                isVisible={
                    isTranslatedMessageVisible ===
                    ETranslatedMessageState.SHOWING
                }
                theme="light"
            >
                <TextMessage
                    className={cx('message__content')}
                    isVisible
                    onClick={hideTranslatedMessage}
                />
            </Snackbar>
        </>
    );
};

export default AviaBookingFormMobile;
