import React, {FC, useCallback, useEffect} from 'react';
import {OnBlur} from 'react-final-form-listeners';
import {useForm} from 'react-final-form';

import {validationInfo} from 'projects/avia/pages/AviaBooking/constants/validation/validationInfo';
import {EFieldName} from 'components/BookingPassengerForm/constants/fieldNames';

import {EAgeCategory} from 'projects/avia/pages/AviaBooking/types/AgeCategoryType';
import {EFormGroup} from 'projects/avia/pages/AviaBooking/types/IBookingFormData';
import {
    IAutoCompleteDocument,
    ISuggestDocument,
} from 'components/BookingPassengerForm/types/IDocument';
import {FlightType} from 'projects/avia/pages/AviaBooking/types/FlightType';
import {IPassengerWithDocumentsDTO} from 'server/api/TravelersApi/types/IPassengerDTO';
import {ICountry} from 'types/common/ICountry';
import {EAviaLoyaltyCardType} from 'types/avia/book/EAviaLoyaltyCardType';

import {IWithQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {getFilteredDocuments} from 'projects/avia/pages/AviaBooking/lib/getFilteredDocuments';
import {translate} from 'utilities/strings/translate';
import setFormValues from 'components/BookingPassengerForm/utilities/setFormValues';
import isDocumentTranslated from '../../utilities/isDocumentTranslated';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18Block from 'i18n/avia-AviaBooking-BookingPassengerWizzard';

import Form from 'components/Form/Form';
import BookingPassengerForm from 'components/BookingPassengerForm/BookingPassengerForm';
import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import Separator from 'components/Separator/Separator';
import {DefaultSuggestOption} from 'components/Suggest/components/SuggestList/components/DefaultSuggestOption/DefaultSuggestOption';

import LoyaltyCard from '../LoyaltyCard/LoyaltyCard';

interface IAviaBookingPassengerFormProps {
    type: EAgeCategory;
    index: number;
    flightType: FlightType;
    formQaAttributes: IWithQaAttributes;
    disabled?: boolean;
    message?: React.ReactNode;
    passengers?: IPassengerWithDocumentsDTO[];
    countries: ICountry[];
    allowedLoyaltyPrograms: EAviaLoyaltyCardType[];
    showTranslatedMessage: (index: number, value?: boolean) => void;
}

export const AviaBookingPassengerForm: FC<IAviaBookingPassengerFormProps> = ({
    type,
    index,
    flightType,
    formQaAttributes,
    disabled,
    message,
    passengers,
    countries,
    allowedLoyaltyPrograms,
    showTranslatedMessage,
}) => {
    const formApi = useForm();
    const deviceType = useDeviceType();

    const title = i18Block[type]({
        index: index + 1,
    });

    const groupId = `${EFormGroup.passengers}.${index}`;

    const lastNameFieldName = `${groupId}.${EFieldName.lastName}`;
    const firstNameFieldName = `${groupId}.${EFieldName.firstName}`;
    const patronymicNameFieldName = `${groupId}.${EFieldName.patronymicName}`;

    const availableDocuments = getFilteredDocuments(flightType, type);

    const onNameFieldsBlur = useCallback(
        (fieldName: string) => {
            const value = formApi.getFieldState(fieldName)?.value;
            const translatedValue = translate(value);

            if (value !== translatedValue) {
                formApi.change(fieldName, translatedValue.trim());
                showTranslatedMessage(index);
            }
        },
        [formApi, index, showTranslatedMessage],
    );

    // на случай если в инит данных есть имена на кирилице
    useEffect(() => {
        onNameFieldsBlur(lastNameFieldName);
        onNameFieldsBlur(firstNameFieldName);
        onNameFieldsBlur(patronymicNameFieldName);
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, []);

    const onLastFieldsBlur = useCallback(
        () => onNameFieldsBlur(lastNameFieldName),
        [onNameFieldsBlur, lastNameFieldName],
    );
    const onFirstFieldsBlur = useCallback(
        () => onNameFieldsBlur(firstNameFieldName),
        [onNameFieldsBlur, firstNameFieldName],
    );
    const onPatronymicFieldsBlur = useCallback(
        () => onNameFieldsBlur(patronymicNameFieldName),
        [onNameFieldsBlur, patronymicNameFieldName],
    );

    const handleDocumentSelect = useCallback(
        (document: IAutoCompleteDocument): void => {
            setFormValues(formApi, groupId, document, availableDocuments, {
                hasPatronymic: true,
            });

            showTranslatedMessage(index, isDocumentTranslated(document));
        },
        [availableDocuments, formApi, groupId, index, showTranslatedMessage],
    );

    const renderSuggestOption = useCallback(
        (document: ISuggestDocument) => (
            <DefaultSuggestOption
                option={{
                    title: translate(document.title),
                    comment: document.comment,
                }}
            />
        ),
        [],
    );

    return (
        <Form.FieldGroup groupId={groupId}>
            <BookingLayout.Card between={9}>
                <BookingPassengerForm
                    deviceType={deviceType}
                    title={title}
                    validationInfo={validationInfo}
                    message={message}
                    availableDocuments={availableDocuments}
                    passengers={passengers}
                    countries={countries}
                    disabled={disabled}
                    isLatinName
                    hasPatronymic
                    hasDocumentDate
                    handleSuggestSelect={handleDocumentSelect}
                    handleIntentSelect={handleDocumentSelect}
                    renderSuggestOption={renderSuggestOption}
                    keepInvalidFields
                    {...formQaAttributes}
                />

                <OnBlur name={lastNameFieldName}>{onLastFieldsBlur}</OnBlur>
                <OnBlur name={firstNameFieldName}>{onFirstFieldsBlur}</OnBlur>
                <OnBlur name={patronymicNameFieldName}>
                    {onPatronymicFieldsBlur}
                </OnBlur>

                {allowedLoyaltyPrograms.length && (
                    <>
                        {deviceType.isMobile && <Separator margin="4" />}

                        <LoyaltyCard
                            allowedLoyaltyPrograms={allowedLoyaltyPrograms}
                            disabled={disabled}
                        />
                    </>
                )}
            </BookingLayout.Card>
        </Form.FieldGroup>
    );
};
