import React, {
    ReactNode,
    useEffect,
    useContext,
    useMemo,
    useCallback,
    useRef,
} from 'react';
import {useField} from 'react-final-form';

import {AVIA_LOYALTY_CARD_TYPE_TO_TITLE_MAP} from 'projects/avia/pages/AviaBooking/components/AviaBookingForm/components/LoyaltyCard/constants';

import {EAviaLoyaltyCardType} from 'types/avia/book/EAviaLoyaltyCardType';
import {
    EAviaLoyaltyCardFieldName,
    EAviaPassengerFieldName,
} from 'projects/avia/pages/AviaBooking/types/IBookingFormPassengersData';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {useToggle} from 'utilities/hooks/useToggle';

import * as i18nBlock from 'i18n/aviaBookingForm';

import RotatingArrowIcon from 'components/RotatingArrowIcon/RotatingArrowIcon';
import LinkButton from 'components/LinkButton/LinkButton';
import AutoHeightAnimateControl from 'components/AutoHeightAnimateControl/AutoHeightAnimateControl';
import Form from 'components/Form/Form';
import FormField from 'components/FormField/FormField';
import Select from 'components/Select/Select';
import Box from 'components/Box/Box';
import {useToggleHandler} from 'projects/avia/pages/AviaBooking/components/AviaBookingForm/components/LoyaltyCard/hooks/useToggleHandler';
import BookingInput from 'components/BookingPassengerForm/components/BookingInput/BookingInput';
import {TControlRenderFunc} from 'components/FormField/components/Field/Field';

import cx from './LoyaltyCard.scss';

interface IBonusCardProps {
    allowedLoyaltyPrograms: EAviaLoyaltyCardType[];
    disabled: boolean | undefined;
}

const LoyaltyCard: React.FC<IBonusCardProps> = props => {
    const {allowedLoyaltyPrograms, disabled = false} = props;

    const deviceType = useDeviceType();
    const [isExpanded, toggle] = useToggle(false);
    const toggleHandler = useToggleHandler(isExpanded, toggle);
    const formGroupId = useContext(Form.FieldGroupContext);

    const field = useField(
        `${formGroupId}.${EAviaPassengerFieldName.loyaltyCard}.${EAviaLoyaltyCardFieldName.number}`,
        {
            subscription: {data: true},
        },
    );
    const submitAmount = field.meta.data?.submitAmount;

    useEffect(() => {
        if (submitAmount && toggleHandler) {
            toggleHandler();
        }
    }, [submitAmount]);

    const options = useMemo(
        () =>
            allowedLoyaltyPrograms.map(card => ({
                value: card,
                data: AVIA_LOYALTY_CARD_TYPE_TO_TITLE_MAP[card],
            })),
        [allowedLoyaltyPrograms],
    );

    const ref = useRef(null);

    const selectComponent: TControlRenderFunc = useCallback(
        ({input}): ReactNode => (
            <Select
                {...input}
                fixPopupInSpring
                scope={ref}
                id={input.name}
                options={options}
                disabled={!isExpanded || disabled}
                width="max"
                theme="outlined"
                size="l"
            />
        ),
        [disabled, isExpanded, options],
    );

    return (
        <Box className={cx('root', deviceMods('root', deviceType))} ref={ref}>
            <LinkButton className={cx('toggleLink')} onClick={toggleHandler}>
                {i18nBlock.loyaltyCardGroupTitle()}

                <RotatingArrowIcon
                    className={cx('toggleLinkIcon')}
                    rotated={isExpanded}
                />
            </LinkButton>

            <Form.FieldGroup groupId={EAviaPassengerFieldName.loyaltyCard}>
                <AutoHeightAnimateControl canShowComponent={isExpanded}>
                    <Box
                        className={cx('cardInfoWrapper')}
                        between={deviceType.isMobile ? 3 : 0}
                    >
                        <FormField
                            name={EAviaLoyaltyCardFieldName.type}
                            className={cx('type')}
                            title={i18nBlock.loyaltyCardTypeTitle()}
                            deviceType={deviceType}
                            control={selectComponent}
                        />

                        <BookingInput
                            name={EAviaLoyaltyCardFieldName.number}
                            className={cx('number')}
                            title={i18nBlock.loyaltyCardNumberTitle()}
                            deviceType={deviceType}
                            disabled={!isExpanded || disabled}
                        />
                    </Box>
                </AutoHeightAnimateControl>
            </Form.FieldGroup>
        </Box>
    );
};

export default React.memo(LoyaltyCard);
