import {IBookingSegmentModel} from 'server/services/AviaBookingService/variants/types';
import {EAgeCategory} from '../../../types/AgeCategoryType';
import {IPassengerWithDocumentsDTO} from 'server/api/TravelersApi/types/IPassengerDTO';
import TAgeCategoryInfo from 'projects/avia/types/TAgeCategoryInfo';

import {ROBOT} from 'utilities/dateUtils/formats';
import getAgeGroup from 'projects/avia/utilities/getAgeGroup';
import {getBoardingAge} from 'utilities/dateUtils/getBoardingAge';

import {getFlightInfo} from 'projects/avia/pages/AviaBookingCreateOrder/AviaBookingCreateOrder';

/*
    Детям и младенцам в общем случае нельзя покупать взрослое место.
    Детские интенты можно показывать только для детей; младенческие - можно для детей и младенцев.
*/
const AGE_CATEGORIES: TAgeCategoryInfo = {
    [EAgeCategory.infant]: {
        minAge: 0,
        maxAge: 2,
    },
    [EAgeCategory.child]: {
        minAge: 0,
        maxAge: 12,
    },
    [EAgeCategory.adult]: {
        minAge: 12,
        maxAge: 200,
    },
};

/**
 * Функция возвращает отфильтрованных по возрастной группе пассажиров
 *
 * @param passengers - пассажиры
 * @param type - возрастная группа для фильтрации
 * @param departureDate - дата отправления
 */
const filterPassengersWithDocumentsByType = ({
    passengersWithDocuments,
    type,
    departureDate,
}: {
    passengersWithDocuments: IPassengerWithDocumentsDTO[];
    type: EAgeCategory;
    departureDate: string;
}): IPassengerWithDocumentsDTO[] => {
    return passengersWithDocuments.filter(passenger => {
        const boardingAge = getBoardingAge(
            passenger.birth_date,
            departureDate,
            ROBOT,
        );

        const passengerAgeGroups = getAgeGroup(boardingAge, AGE_CATEGORIES);

        return passengerAgeGroups && passengerAgeGroups.includes(type);
    });
};

export const groupPassengersByType = ({
    types,
    segments,
    passengersWithDocuments,
}: {
    segments: IBookingSegmentModel[];
    types: EAgeCategory[];
    passengersWithDocuments: IPassengerWithDocumentsDTO[];
}): Partial<Record<EAgeCategory, IPassengerWithDocumentsDTO[]>> => {
    const passengersGroupedByType: Partial<
        Record<EAgeCategory, IPassengerWithDocumentsDTO[]>
    > = {};

    const flightInfo = getFlightInfo(segments);
    const departureDate =
        flightInfo?.lastFlightDepartureDate ||
        flightInfo?.firstFlightDepartureDate ||
        '';

    types.forEach(type => {
        passengersGroupedByType[type] = filterPassengersWithDocumentsByType({
            passengersWithDocuments,
            type,
            departureDate,
        });
    });

    return passengersGroupedByType;
};
