import React, {useCallback} from 'react';

import {EAppActions} from 'constants/platforms/TPlatforms';

import {IBookingVariantsModelType} from 'server/services/AviaBookingService/variants/types';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {aviaURLs} from 'projects/avia/lib/urls';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';

import * as i18nBlock from 'i18n/avia-AviaBooking-BookingBreadcrumbs';

import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import SupportPhone from 'projects/avia/pages/AviaBooking/components/SupportPhone/SupportPhone';

import {useCoordinator, usePlatform} from 'contexts/PlatformContext';

import cx from './BookingBreadcrumbs.scss';

export enum EBookingBreadcrumbsStep {
    FORM,
    PAYMENT,
}

interface IBookingBreadcrumbsProps extends IWithQaAttributes {
    variants: IBookingVariantsModelType;
    activeStep: EBookingBreadcrumbsStep;
    variantToken: string;
    children?: never;
}

export const BookingBreadcrumbs: React.FC<IBookingBreadcrumbsProps> = props => {
    const {activeStep, variants, variantToken} = props;
    const {isDesktop} = useDeviceType();

    const {isTravelApp} = usePlatform();
    const coordinator = useCoordinator();

    const handleSearchBreadcrumbClick: React.MouseEventHandler = useCallback(
        e => {
            if (isTravelApp) {
                coordinator.doAction(EAppActions.NAVIGATE_TO_SEARCH_AVIA_PAGE);

                e.preventDefault();
            }
        },
        [coordinator, isTravelApp],
    );
    const handleOrderBreadcrumbClick: React.MouseEventHandler = useCallback(
        e => {
            if (isTravelApp) {
                coordinator.doAction(EAppActions.NAVIGATE_TO_VARIANT_AVIA_PAGE);

                e.preventDefault();
            }
        },
        [coordinator, isTravelApp],
    );

    return (
        <BookingLayout.Breadcrumbs
            additional={
                isDesktop && (
                    <BookingLayout.Breadcrumbs.SupportItem>
                        <SupportPhone />
                    </BookingLayout.Breadcrumbs.SupportItem>
                )
            }
            {...prepareQaAttributes(props)}
        >
            <BookingLayout.Breadcrumbs.Item
                className={cx('item')}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'item-search',
                })}
                to={aviaURLs.getSearchResultsUrlByQid(variants.qid)}
                onClick={handleSearchBreadcrumbClick}
            >
                {i18nBlock.searchDashLabel()}
            </BookingLayout.Breadcrumbs.Item>
            <BookingLayout.Breadcrumbs.Item
                className={cx('item')}
                /*
                Пока оставил url
                Страница заказа не умеет открываться через переход в SPA: не заполнен контекст
                https://st.yandex-team.ru/TRAVELFRONT-4150#5f88318096ffcc6eb4de6e3d
                */
                url={aviaURLs.getOrderUrlByQidAndRouteFlights(
                    variants.qid,
                    variants.flightsByRoute,
                )}
                onClick={handleOrderBreadcrumbClick}
                {...prepareQaAttributes({parent: props, current: 'item-order'})}
            >
                {i18nBlock.orderDashLabel()}
            </BookingLayout.Breadcrumbs.Item>
            <BookingLayout.Breadcrumbs.Item
                active={activeStep === EBookingBreadcrumbsStep.FORM}
                className={cx('item')}
                key="form"
                to={aviaURLs.getBookingFormUrl(
                    variantToken,
                    getQueryByLocation(document.location) as Record<
                        string,
                        string
                    >,
                )}
                {...prepareQaAttributes({parent: props, current: 'item-form'})}
            >
                {i18nBlock.formDashLabel()}
            </BookingLayout.Breadcrumbs.Item>
            <BookingLayout.Breadcrumbs.Item
                active={activeStep === EBookingBreadcrumbsStep.PAYMENT}
                className={cx('item')}
                key="payment"
                disabled={activeStep !== EBookingBreadcrumbsStep.PAYMENT}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'item-payment',
                })}
            >
                {i18nBlock.paymentDashLabel()}
            </BookingLayout.Breadcrumbs.Item>
        </BookingLayout.Breadcrumbs>
    );
};
