import React, {useRef} from 'react';

import {IBookingVariantsModelType} from 'server/services/AviaBookingService/variants/types';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import IPrice from 'utilities/currency/PriceInterface';

import ErrorModal from 'components/ErrorModal/ErrorModal';

import {useCoordinator, usePlatform} from 'contexts/PlatformContext';

import getModalConfig from './configs/getModalConfig';

import cx from './BookingErrorModal.scss';

export enum EBookingModalErrorType {
    Unknown = 'Unknown',
    PriceChanged = 'PriceChanged',
    TariffNotAvailable = 'TariffNotAvailable',
    SameNameRestriction = 'SameNameRestriction',
}

interface IBookingErrorModalProps {
    errorType?: EBookingModalErrorType;
    onClose?: () => void;
    variants?: IBookingVariantsModelType | null;
    newPrice?: IPrice | null;
    variantToken: string;
}

export const BookingErrorModal: React.FC<IBookingErrorModalProps> = ({
    errorType,
    onClose,
    variants,
    newPrice,
    variantToken,
}) => {
    const {isMobile} = useDeviceType();
    const actualErrorTypeRef = useRef<EBookingModalErrorType | undefined>(
        errorType,
    );

    const {isTravelApp} = usePlatform();
    const coordinator = useCoordinator();

    if (errorType) {
        actualErrorTypeRef.current = errorType;
    }

    if (!variants) {
        return null;
    }

    const actualErrorType = actualErrorTypeRef.current;

    if (!actualErrorType) {
        return null;
    }

    const {title, content, primaryAction, secondaryAction} = getModalConfig({
        variantToken,
        errorType: actualErrorType,
        variants,
        newPrice,
        onClose,
        isMobile,
        isTravelApp,
        coordinator,
    });

    return (
        <ErrorModal
            title={title}
            isVisible={Boolean(errorType)}
            primaryAction={primaryAction}
            secondaryAction={secondaryAction}
            {...prepareQaAttributes({
                current: errorType,
                parent: 'aviaBookingModalError',
            })}
        >
            <div className={cx('root')}>{content}</div>
        </ErrorModal>
    );
};
