import find from 'lodash/find';
import get from 'lodash/get';

import {EAppActions} from 'constants/platforms/TPlatforms';

import {IBookingVariantsModelType} from 'server/services/AviaBookingService/variants/types';

import IPrice from 'utilities/currency/PriceInterface';
import {aviaURLs} from 'projects/avia/lib/urls';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBlock from 'i18n/avia-AviaBooking-error-modals';

import Price from 'components/Price/Price';

import {ICoordinator} from 'contexts/PlatformContext';

import IModalErrorConfig from './IModalErrorConfig';

export default function getPriceChangeConfig({
    variantToken,
    variants,
    newPrice,
    isTravelApp,
    coordinator,
    onClose,
}: {
    variantToken: string;
    variants: IBookingVariantsModelType;
    newPrice?: IPrice | null;
    isTravelApp: boolean;
    coordinator: ICoordinator;
    onClose?: () => void;
}): IModalErrorConfig {
    const selectedVariant =
        find(variants.allVariants, {id: variantToken}) ||
        variants.allVariants[0];

    const tariffName = get(
        selectedVariant,
        'segments[0].flights[0].fareTerms.tariffGroupName',
    );
    const price = newPrice || variants.priceInfo.first_check_price;

    return {
        title: i18nBlock.priceDashChangeDashTitle(),
        content: insertJSXIntoKey(i18nBlock.priceDashChangeDashDescription)({
            tariff: tariffName,
            price: (
                <Price
                    key="price"
                    value={price.value}
                    currency={price.currency}
                />
            ),
        }),
        primaryAction: {
            type: 'button',
            title: i18nBlock.priceDashChangeDashButtonDashContinue(),
            handler: onClose,
        },
        secondaryAction: isTravelApp
            ? {
                  type: 'button',
                  title: i18nBlock.priceDashChangeDashButtonDashBack(),
                  handler(): void {
                      coordinator.doAction(
                          EAppActions.NAVIGATE_TO_SEARCH_AVIA_PAGE,
                      );
                  },
              }
            : {
                  type: 'button-link',
                  title: i18nBlock.priceDashChangeDashButtonDashBack(),
                  props: {
                      to: aviaURLs.getSearchResultsUrlByQid(variants.qid),
                  },
              },
    };
}
