import React from 'react';

import {IBookingFlightModel} from 'server/services/AviaBookingService/variants/types';

import {parseDate} from 'utilities/dateUtils';
import {ROBOT} from 'utilities/dateUtils/formats';
import {useMobile} from 'utilities/hooks/useMobile';
import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Box from 'components/Box/Box';
import {AviaFlightTimings} from 'projects/avia/components/AviaFlightTimings/AviaFlightTimings';
import {PassengerExperience} from 'projects/avia/components/PassengerExperience/PassengerExperience';
import AirlineTitle from './components/AirlineTitle/AirlineTitle';
import FareTermsButton from './components/FareTermsButton/FareTermsButton';
import BaggageFormatter from 'projects/avia/components/BaggageFormatter/BaggageFormatter';
import PassengerExperienceFormatter from './components/PassengerExperienceFormatter/PassengerExperienceFormatter';
import BaggageIcon from 'icons/16/Baggage';
import HandLuggageIcon from 'icons/16/HandLuggage';

import cx from './BookingFlightInfo.scss';

interface IBookingFlightInfoProps extends IWithQaAttributes {
    flight: IBookingFlightModel;
    fareModalBackText?: string;
    showDate: boolean;
    withBaggage: boolean;
}

const BookingFlightInfo: React.FC<IBookingFlightInfoProps> = props => {
    const {flight, fareModalBackText, showDate, withBaggage} = props;
    const {
        departureSettlement,
        arrivalSettlement,
        departureDate,
        arrivalDate,
        departureAirport,
        arrivalAirport,
        company,
        operatingCompany,
        flightNumber,
        fareTerms,
    } = flight;

    const isMobile = useMobile();

    const carryOn = fareTerms.terms.carryOn;
    const baggage = fareTerms.terms.baggage;

    return (
        <Box {...prepareQaAttributes(props)} className={cx('root', {isMobile})}>
            <div className={cx('row', 'content-row')}>
                <Box className={cx('flex-1')} between="3">
                    <AirlineTitle
                        {...prepareQaAttributes(props)}
                        company={company}
                        operatingCompany={operatingCompany}
                        flightNumber={flightNumber}
                    />
                    <Box className={cx('passenger-experience')}>
                        <PassengerExperience
                            iata={operatingCompany.iataCode}
                            flightNumber={flightNumber}
                            when={parseDate(departureDate).format(ROBOT)}
                        >
                            {({
                                passengerExperience,
                                isLoading,
                            }): React.ReactNode =>
                                (isLoading || passengerExperience) && (
                                    <div
                                        className={cx(isLoading && '_loading')}
                                    >
                                        {passengerExperience && (
                                            <PassengerExperienceFormatter
                                                {...prepareQaAttributes(props)}
                                                {...passengerExperience}
                                            />
                                        )}
                                    </div>
                                )
                            }
                        </PassengerExperience>
                        {withBaggage && (
                            <>
                                <Box>
                                    <FareTermsButton
                                        {...fareTerms}
                                        fareModalBackText={fareModalBackText}
                                        {...prepareQaAttributes({
                                            parent: props,
                                            current: 'fareTermsButton',
                                        })}
                                    />
                                </Box>
                                <Box between={4} inline>
                                    <BaggageFormatter
                                        {...prepareQaAttributes({
                                            parent: props,
                                            current: 'carry-on',
                                        })}
                                        places={carryOn?.places}
                                        weight={carryOn?.weight}
                                        size={carryOn?.size}
                                        icon={<HandLuggageIcon />}
                                    />
                                    <BaggageFormatter
                                        {...prepareQaAttributes({
                                            parent: props,
                                            current: 'baggage',
                                        })}
                                        showNoBaggage
                                        places={baggage?.places}
                                        weight={baggage?.weight}
                                        icon={<BaggageIcon />}
                                    />
                                </Box>
                            </>
                        )}
                    </Box>
                </Box>
                <Box className={cx('flex-1')} between={isMobile ? 0 : 3}>
                    <AviaFlightTimings
                        {...prepareQaAttributes(props)}
                        departureDate={departureDate}
                        arrivalDate={arrivalDate}
                        departureAirportTimezoneOffset={
                            departureAirport.timeZoneOffset
                        }
                        arrivalAirportTimezoneOffset={
                            arrivalAirport.timeZoneOffset
                        }
                        showDate={showDate}
                    />
                    <div className={cx('row')}>
                        <div className={cx('flex-1', 'ta-left')}>
                            <div
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'fromCity',
                                })}
                            >
                                <strong>{departureSettlement.title}</strong>
                            </div>
                            <div>
                                <span
                                    {...prepareQaAttributes({
                                        parent: props,
                                        current: 'fromAirport',
                                    })}
                                >
                                    {departureAirport.title}
                                </span>
                                &nbsp;{' '}
                                <span
                                    {...prepareQaAttributes({
                                        parent: props,
                                        current: 'fromAirportCode',
                                    })}
                                >
                                    {departureAirport.code}
                                </span>
                            </div>
                        </div>
                        <div className={cx('flex-1', 'ta-right')}>
                            <div
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'toCity',
                                })}
                            >
                                <strong>{arrivalSettlement.title}</strong>
                            </div>
                            <div>
                                <span
                                    {...prepareQaAttributes({
                                        parent: props,
                                        current: 'toAirport',
                                    })}
                                >
                                    {arrivalAirport.title}
                                </span>
                                &nbsp;{' '}
                                <span
                                    {...prepareQaAttributes({
                                        parent: props,
                                        current: 'toAirportCode',
                                    })}
                                >
                                    {arrivalAirport.code}
                                </span>
                            </div>
                        </div>
                    </div>
                </Box>
            </div>
        </Box>
    );
};

export default BookingFlightInfo;
