import React from 'react';

import {
    IVariantPriceInfoDTO,
    ICategoryPriceDTO,
} from 'server/api/AviaBookingApi/types/IBookingVariantsResponse';

import {useMobile} from 'utilities/hooks/useMobile';
import {useToggle} from 'utilities/hooks/useToggle';
import {
    formatCountOfAdults,
    formatCountOfChildren,
    formatCountOfInfants,
} from 'utilities/formatCount';
import IPrice from 'utilities/currency/PriceInterface';
import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as localized from 'i18n/avia-AviaBooking-BookingPriceInfo';

import Box from 'components/Box/Box';
import Price from 'components/Price/Price';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import RotatingArrowIcon from 'components/RotatingArrowIcon/RotatingArrowIcon';
import AutoHeightAnimateControl from 'components/AutoHeightAnimateControl/AutoHeightAnimateControl';

import cx from './BookingPriceInfo.scss';

interface IBookingPriceInfoProps extends IWithQaAttributes {
    priceInfo: IVariantPriceInfoDTO;
    oldPrice?: IPrice;
}

export const BookingPriceInfo: React.FC<IBookingPriceInfoProps> = props => {
    const {priceInfo, oldPrice} = props;
    const [detailedPricesAreShown, toggleDetailedPrices] = useToggle(false);
    const isMobile = useMobile();

    const {total, categoryPrices} = priceInfo;

    const passengerIsAlone = categoryPrices.length === 1;

    const totalLabelNode = passengerIsAlone ? (
        <div>{localized.total()}</div>
    ) : (
        <button
            onClick={toggleDetailedPrices}
            className={cx('total-button')}
            type="button"
        >
            <TextWithIcon
                size={isMobile ? 'm' : 'xl'}
                text={localized.total()}
                iconRight={RotatingArrowIcon}
                iconRightProps={{
                    rotated: detailedPricesAreShown,
                    width: isMobile ? 12 : 16,
                    height: isMobile ? 12 : 16,
                }}
            />
        </button>
    );

    return (
        <section
            className={cx('root', isMobile && 'root_touch')}
            {...prepareQaAttributes(props)}
        >
            <div className={cx('title')}>
                {totalLabelNode}
                <Price
                    currency={total.currency}
                    value={total.value}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'total-price',
                    })}
                />
            </div>
            {oldPrice && (
                <>
                    <Box below="2" className={cx('old-price')}>
                        <div className={cx('old-price__wrapper')}>
                            <Price
                                currency={oldPrice.currency}
                                value={oldPrice.value}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'oldPrice',
                                })}
                            />
                        </div>
                    </Box>
                    <Box
                        textSize="m"
                        textColor="alert"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'updatedPriceAlert',
                        })}
                    >
                        {localized.updatedDashPrice()}
                    </Box>
                </>
            )}
            <AutoHeightAnimateControl canShowComponent={detailedPricesAreShown}>
                <ul className={cx('price-details-list')}>
                    {categoryPrices
                        .slice()
                        .sort(categoryPricesComparator)
                        .map((categoryPrice, idx) => (
                            <li
                                key={idx}
                                className={cx('price-details-list-item')}
                            >
                                <span>
                                    {formatPassengerCount(
                                        categoryPrice.quantity,
                                        categoryPrice.passengerCategory,
                                    )}
                                </span>
                                <span>
                                    <Price
                                        currency={categoryPrice.total.currency}
                                        value={categoryPrice.total.value}
                                    />
                                </span>
                            </li>
                        ))}
                </ul>
            </AutoHeightAnimateControl>
        </section>
    );
};

function formatPassengerCount(count: number, category: string): string {
    switch (category) {
        case 'adult':
            return formatCountOfAdults(count);
        case 'child':
            return formatCountOfChildren(count);
        case 'infant':
            return formatCountOfInfants(count);
        default:
            return '';
    }
}

function categoryPricesComparator(
    categoryPriceA: ICategoryPriceDTO,
    categoryPriceB: ICategoryPriceDTO,
): 1 | 0 | -1 {
    return passengerCategoryComparator(
        categoryPriceA.passengerCategory,
        categoryPriceB.passengerCategory,
    );
}

function passengerCategoryComparator(
    categoryA: string,
    categoryB: string,
): 1 | 0 | -1 {
    const weights: Record<string, number> = {
        adult: 100,
        child: 10,
        infant: 1,
    };
    const wA = weights[categoryA] || 0;
    const wB = weights[categoryB] || 0;

    if (wA < wB) {
        return 1;
    }

    if (wA > wB) {
        return -1;
    }

    return 0;
}
