import React from 'react';
import first from 'lodash/first';
import last from 'lodash/last';

import {IBookingSegmentModel} from 'server/services/AviaBookingService/variants/types';

import {formatDate, parseDate} from 'utilities/dateUtils';
import {HUMAN_WITH_WEEK} from 'utilities/dateUtils/formats';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useMobile} from 'utilities/hooks/useMobile';

import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import Box from 'components/Box/Box';
import Intersperse from 'components/Intersperse/Intersperse';

import {TransferInfo} from '../TransferInfo/TransferInfo';
import BookingFlightInfo from '../BookingFlightInfo/BookingFlightInfo';

import cx from './BookingSegmentsInfo.scss';

interface IBookingSegmentInfoProps extends IWithQaAttributes {
    segments: IBookingSegmentModel[];
    withBaggage: boolean;
}

const BookingSegmentsInfo: React.FC<IBookingSegmentInfoProps> = props => {
    const {segments, withBaggage} = props;

    const isMobile = useMobile();

    return (
        <BookingLayout.Snippets below={0}>
            {segments.map((segment, idx) => {
                const firstFlight = segment.flights[0];

                return (
                    <BookingLayout.Card
                        key={idx}
                        className={cx('card', {isMobile})}
                    >
                        <Box between="5">
                            <div className={cx('row', 'title-row')}>
                                <h3 className={cx('title-heading')}>
                                    {getSegmentTitle(segment)}
                                </h3>
                                <span className={cx('title-subheading')}>
                                    {firstFlight &&
                                        formatDate(
                                            firstFlight.departureDate,
                                            HUMAN_WITH_WEEK,
                                        )}
                                </span>
                            </div>
                            <Intersperse
                                separator={(_idx): React.ReactNode => (
                                    <TransferInfo
                                        key={`transfer-${_idx}`}
                                        arrivalFlight={segment.flights[_idx]}
                                        departureFlight={
                                            segment.flights[_idx + 1]
                                        }
                                    />
                                )}
                            >
                                {segment.flights.map(flight => {
                                    return (
                                        <BookingFlightInfo
                                            fareModalBackText={
                                                getSegmentTitle(segment) || ''
                                            }
                                            flight={flight}
                                            key={flight.id}
                                            withBaggage={withBaggage}
                                            showDate={parseDate(
                                                flight.departureDate,
                                            ).isAfter(
                                                firstFlight.departureDate,
                                                'date',
                                            )}
                                            {...prepareQaAttributes({
                                                parent: props,
                                                current: 'flightInfo',
                                                key: flight.id,
                                            })}
                                        />
                                    );
                                })}
                            </Intersperse>
                        </Box>
                    </BookingLayout.Card>
                );
            })}
        </BookingLayout.Snippets>
    );
};

function getSegmentTitle(segment: IBookingSegmentModel): string | null {
    const firstFlight = first(segment.flights);
    const lastFlight = last(segment.flights);

    if (!firstFlight || !lastFlight) {
        return null;
    }

    return `${firstFlight.departureSettlement.title} — ${lastFlight.arrivalSettlement.title}`;
}

export default BookingSegmentsInfo;
