import {FC} from 'react';
import {useSelector} from 'react-redux';

import {ISegmentFare} from 'server/api/AviaBookingApi/types/ISegmentFare';
import {IYandexPlusPromoCampaign} from 'server/api/AviaBookingApi/types/IBookingVariantsResponse';

import {aviaAeroflotPlusSelector} from 'projects/avia/selectors/aviaAeroflotPlusEnabledSelector';

import {
    formatTermAvailability,
    formatTermType,
} from 'projects/avia/lib/flightFareTerms/formatters';
import IPrice from 'utilities/currency/PriceInterface';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Price from 'components/Price/Price';
import Box from 'components/Box/Box';
import Heading from 'components/Heading/Heading';
import BaggageFormatterVerbose from 'projects/avia/components/BaggageFormatterVerbose/BaggageFormatterVerbose';
import Text from 'components/Text/Text';

import cx from './BookingTariffInfo.scss';

export interface IBookingTariffInfoProps {
    fareTerms: ISegmentFare;
    price?: IPrice;
    isInRadio?: boolean;
    oldPrice?: IPrice;
    plusPromo?: IYandexPlusPromoCampaign;
}

const BookingTariffInfo: FC<IBookingTariffInfoProps> = ({
    fareTerms,
    price,
    isInRadio,
    oldPrice,
    plusPromo,
}) => {
    const aviaAeroflotPlus = useSelector(aviaAeroflotPlusSelector);
    const deviceType = useDeviceType();
    const {isMobile} = deviceType;
    const {baggage, carryOn, miles, refundable, changingCarriage} =
        fareTerms.terms;
    const title = fareTerms.tariffGroupName.toLowerCase();

    return (
        <Box between={isInRadio ? 3 : 2}>
            <Box between={1}>
                <Heading
                    level={3}
                    className={cx('title', isMobile && 'title_touch')}
                >
                    <Text
                        tag="div"
                        color={
                            aviaAeroflotPlus && plusPromo?.enabled
                                ? 'plus'
                                : undefined
                        }
                        className={cx('title_label')}
                    >
                        {title}
                    </Text>
                    <div>
                        {price && (
                            <Price
                                value={price.value}
                                currency={price.currency}
                            />
                        )}
                        {oldPrice && (
                            <div className={cx('old-price')}>
                                <div className={cx('old-price__wrapper')}>
                                    <Price
                                        value={oldPrice.value}
                                        currency={oldPrice.currency}
                                    />
                                </div>
                            </div>
                        )}
                    </div>
                </Heading>
            </Box>

            <Box tag="ul" between={2} className={cx('option-list')}>
                <li className={cx('option')}>
                    {formatTermType('carryOn')}:{' '}
                    <BaggageFormatterVerbose showNoBaggage {...carryOn} />
                </li>
                <li className={cx('option')}>
                    {formatTermType('baggage')}:{' '}
                    <BaggageFormatterVerbose
                        showNoBaggage
                        places={baggage.places}
                        weight={baggage.weight}
                    />
                </li>
                <li className={cx('option')}>
                    {formatTermType('miles')}:{' '}
                    {(miles.miles || '').toLowerCase()}
                </li>
                <li className={cx('option')}>
                    {formatTermType('refundable')}:{' '}
                    {formatTermAvailability(
                        refundable.availability,
                    ).toLowerCase()}
                </li>
                <li className={cx('option')}>
                    {formatTermType('changingCarriage')}:{' '}
                    {formatTermAvailability(
                        changingCarriage.availability,
                    ).toLowerCase()}
                </li>
            </Box>
        </Box>
    );
};

export default BookingTariffInfo;
