import React, {useCallback, useMemo} from 'react';
import {Field} from 'react-final-form';
import {useDispatch} from 'react-redux';

import {EFormGroup} from 'projects/avia/pages/AviaBooking/types/IBookingFormData';
import {IBookingVariantModel} from 'server/services/AviaBookingService/variants/types';
import {EFlightInfoFieldName} from '../../types/IBookingFlightInfo';

import {openTariffsModal} from 'reducers/avia/booking/tariffsModalIsOpened/actions';

import IPrice from 'utilities/currency/PriceInterface';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getTariffsSlice from './utilities/getTariffsSlice';

import * as i18nBlockTariffsField from 'i18n/avia-AviaBooking-TariffsField';

import Box from 'components/Box/Box';
import TariffsModal from 'projects/avia/pages/AviaBooking/components/TariffsField/components/TariffsModal/TariffsModal';
import LinkButton from 'components/LinkButton/LinkButton';

import {TariffsRadioGroup} from '../TariffsRadioGroup/TariffsRadioGroup';
import {variantModelToTariffInfoList} from './variantModelToTariffInfoList';

import cx from './TariffsField.scss';

interface ITariffsFieldProps extends IWithQaAttributes {
    variants: IBookingVariantModel[];
    oldPrices?: Record<string, IPrice | undefined>;
    disabled?: boolean;
}

const TariffsField: React.FC<ITariffsFieldProps> = props => {
    const {variants, disabled, oldPrices = {}} = props;

    const dispatch = useDispatch();

    const handleOpenButtonClick = useCallback(() => {
        dispatch(openTariffsModal());
    }, [dispatch]);

    const tariffs = useMemo(() => {
        return variantModelToTariffInfoList(variants);
    }, [variants]);

    return (
        <Box between={5} {...prepareQaAttributes(props)}>
            <Field
                subscription={{value: true}}
                name={`${EFormGroup.flightInfo}.${EFlightInfoFieldName.variantToken}`}
                type="text"
            >
                {({input}): React.ReactNode => {
                    const variantToken = input.value;

                    return (
                        <>
                            <TariffsRadioGroup
                                value={variantToken}
                                disabled={disabled}
                                tariffs={getTariffsSlice(variantToken, tariffs)}
                                oldPrices={oldPrices}
                                onChange={input.onChange}
                            />
                            <TariffsModal
                                variants={variants}
                                variantToken={variantToken}
                                onChange={input.onChange}
                            />
                        </>
                    );
                }}
            </Field>
            <LinkButton
                className={cx('showAllTariffs')}
                onClick={handleOpenButtonClick}
                disabled={disabled}
            >
                {i18nBlockTariffsField.showDashAllDashTariffs()}
            </LinkButton>
        </Box>
    );
};

export default TariffsField;
