import React, {useCallback, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {IVariantForFlightTariff} from 'projects/avia/components/VariantsTariffTable/types/tariffTypes';

import {closeTariffsModal} from 'reducers/avia/booking/tariffsModalIsOpened/actions';

import tariffsModalIsOpenedSelector from 'selectors/avia/booking/tariffsModalIsOpenedSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlockTariffsModal from 'i18n/avia-AviaBooking-TariffsModal';

import Box from 'components/Box/Box';
import {VariantsTariffTable} from 'projects/avia/components/VariantsTariffTable/VariantsTariffTable';
import ModalWithBackButton from 'components/ModalWithBackButton/ModalWithBackButton';
import Modal from 'components/Modal/Modal';
import FullFareRulesLink from 'projects/avia/pages/AviaBooking/components/TariffsField/components/TariffsModal/components/FullFareRulesLink/FullFareRulesLink';

import cx from './TariffsModal.scss';

const ROOT_QA = 'tariffsModal';

interface ITariffsModalProps {
    variants: IVariantForFlightTariff[];
    variantToken?: string;
    onChange?(token: string): void;
}

const TariffsModal: React.FC<ITariffsModalProps> = props => {
    const {variants, variantToken, onChange} = props;

    const deviceType = useDeviceType();

    const dispatch = useDispatch();
    const tariffsModalIsOpened = useSelector(tariffsModalIsOpenedSelector);

    const handleClose = useCallback(() => {
        dispatch(closeTariffsModal());
    }, [dispatch]);

    const handleChange = useMemo(() => {
        if (!onChange) {
            return undefined;
        }

        return (token: string): void => {
            onChange(token);

            handleClose();
        };
    }, [onChange, handleClose]);

    const content = useMemo(() => {
        return (
            <div className={cx('tariffsModalContent')}>
                <Box below={10}>
                    <VariantsTariffTable
                        variants={variants}
                        variantToken={variantToken}
                        onChange={handleChange}
                        {...prepareQaAttributes({
                            parent: ROOT_QA,
                            current: 'table',
                        })}
                    />
                </Box>
                <Box inset={deviceType.isMobile ? 4 : 0}>
                    <FullFareRulesLink
                        {...prepareQaAttributes({
                            parent: ROOT_QA,
                            current: 'fullFareRulesLink',
                        })}
                    />
                </Box>
            </div>
        );
    }, [variants, variantToken, handleChange, deviceType.isMobile]);

    if (deviceType.isMobile) {
        return (
            <ModalWithBackButton
                isVisible={tariffsModalIsOpened}
                fullScreen
                containerClassName={cx('tariffsModal')}
                onClose={handleClose}
                canRenderBackButton
                backButtonText={i18nBlockTariffsModal.backDashButtonDashTitle()}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'modalWithBackButton',
                })}
            >
                {content}
            </ModalWithBackButton>
        );
    }

    return (
        <Modal
            isVisible={tariffsModalIsOpened}
            onClose={handleClose}
            {...prepareQaAttributes({
                parent: ROOT_QA,
                current: 'modal',
            })}
        >
            <Modal.Content>{content}</Modal.Content>
        </Modal>
    );
};

export default TariffsModal;
