import {IBookingFlightModel} from 'server/services/AviaBookingService/variants/types';

import {parseDate} from 'utilities/dateUtils';
import {isNighttimeTransfer} from 'projects/avia/lib/isNighttimeTransfer';

import * as localized from 'i18n/avia-AviaBooking-TransferInfo';

import Separator from 'components/Separator/Separator';
import {Duration} from 'projects/avia/components/Duration/Duration';
import ConnectionIcon from 'icons/12/Connection';
import NightConnectionIcon from 'icons/12/NightConnection';
import Box from 'components/Box/Box';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';

import cx from './TransferInfo.scss';

interface TransferInfoProps {
    arrivalFlight: IBookingFlightModel;
    departureFlight: IBookingFlightModel;
}

export function TransferInfo({
    arrivalFlight,
    departureFlight,
}: TransferInfoProps) {
    const transferIsNighttime = isNighttimeTransfer({
        departureDate: departureFlight.departureDate,
        arrivalDate: arrivalFlight.arrivalDate,
    });
    const transferWithAirportChange = hasAirportChange(
        arrivalFlight,
        departureFlight,
    );

    const duration = (
        <div className={cx('duration')}>
            —
            <Duration
                duration={Duration.minutes(
                    getTransferDurationInMinutes(
                        arrivalFlight,
                        departureFlight,
                    ),
                )}
            />
        </div>
    );

    return (
        <Box between={2}>
            <Separator />
            <div className={cx('inner')}>
                {transferIsNighttime ? (
                    <span className={cx('warning', 'warning--highlighted')}>
                        <TextWithIcon
                            size="m"
                            text={
                                <>
                                    {localized.nightDashTransferDashIn({
                                        settlement:
                                            arrivalFlight.arrivalSettlement
                                                .phraseIn,
                                    })}{' '}
                                    {duration}
                                </>
                            }
                            iconLeft={NightConnectionIcon}
                            iconSize={12}
                        />
                    </span>
                ) : (
                    <span className={cx('warning')}>
                        {localized.transferDashIn({
                            settlement:
                                arrivalFlight.arrivalSettlement.phraseIn,
                        })}{' '}
                        {duration}
                    </span>
                )}
                {transferWithAirportChange && (
                    <span className={cx('warning')}>
                        <TextWithIcon
                            size="m"
                            text={localized.airportDashChange()}
                            iconLeft={ConnectionIcon}
                        />
                    </span>
                )}
            </div>
            <Separator />
        </Box>
    );
}

function getTransferDurationInMinutes(
    arrivalFlight: IBookingFlightModel,
    departureFlight: IBookingFlightModel,
) {
    const departureDateTime = parseDate(departureFlight.departureDate);
    const arrivalDateTime = parseDate(arrivalFlight.arrivalDate);

    return departureDateTime.diff(arrivalDateTime, 'minutes');
}

function hasAirportChange(
    arrivalFlight: IBookingFlightModel,
    departureFlight: IBookingFlightModel,
) {
    const arrivalAirport = arrivalFlight.arrivalAirport.id;
    const departureAirport = departureFlight.departureAirport.id;

    return arrivalAirport !== departureAirport;
}
