import {intersperse} from './intersperse';

const FIELD_PROP_SEPARATOR = '.';

export class ArrayFieldPrefix {
    static fromString(fieldName: string): ArrayFieldPrefix {
        const split = fieldName.split(FIELD_PROP_SEPARATOR);

        if (split.length <= 1) {
            throw new TypeError(
                'Invalid string format: no field name or field index',
            );
        }

        const index = Number(split.pop());

        if (isNaN(index)) {
            throw new TypeError('Invalid string format: no field index');
        }

        const name = split.join(FIELD_PROP_SEPARATOR);

        if (!name) {
            throw new TypeError('Invalid string format: no field name');
        }

        return new ArrayFieldPrefix(name, index);
    }

    /**
     * Часть имени поля стоящая перед последней точкой
     */
    readonly name: string;
    /**
     * Индекс поля в списке полей
     */
    readonly index: number;

    constructor(name: string, index: number) {
        this.name = name;
        this.index = index;
    }

    setIndex(index: number) {
        return new ArrayFieldPrefix(this.name, index);
    }

    toString() {
        return `${this.name}${FIELD_PROP_SEPARATOR}${this.index}`;
    }

    getFieldName(field: string): string {
        return intersperse(
            [this.name, this.index, field],
            FIELD_PROP_SEPARATOR,
        ).join('');
    }
}
