import first from 'lodash/first';

import {RU_CITIZENSHIP_CODE2} from 'constants/document/citizenship';

import {FlightType} from '../types/FlightType';
import {IPassengers} from '../types/IPassengers';
import {EAgeCategory} from '../types/AgeCategoryType';
import {
    IBookingFormPassengersData,
    IFormLoyaltyCard,
} from '../types/IBookingFormPassengersData';
import {EAviaLoyaltyCardType} from 'types/avia/book/EAviaLoyaltyCardType';

import {EDocumentType} from 'server/api/TravelersApi/enums/EDocumentType';

import {getFilteredDocuments} from './getFilteredDocuments';

interface ICreateBookingPassengersOptions {
    flightType: FlightType;
    allowedLoyaltyPrograms: EAviaLoyaltyCardType[];
    middleNameDisabled: boolean;
}

function getInitialLoyaltyCard(
    storedLoyaltyCard: IFormLoyaltyCard | undefined,
    allowedLoyaltyPrograms: EAviaLoyaltyCardType[],
): IFormLoyaltyCard {
    if (
        storedLoyaltyCard?.type &&
        allowedLoyaltyPrograms.includes(storedLoyaltyCard.type)
    ) {
        return storedLoyaltyCard;
    }

    return {
        type: first(allowedLoyaltyPrograms),
        number: '',
    };
}

function formStepFactory(
    ageCategory: EAgeCategory,
    storageDocument: IBookingFormPassengersData | undefined,
    options: ICreateBookingPassengersOptions,
): IBookingFormPassengersData {
    const {allowedLoyaltyPrograms, middleNameDisabled, flightType} = options;
    const documents = getFilteredDocuments(flightType, ageCategory);
    const docType = first(documents) || EDocumentType.RU_NATIONAL_PASSPORT;
    const isValidStorageDocument = storageDocument?.documentType
        ? documents.includes(storageDocument.documentType)
        : false;

    const documentType = isValidStorageDocument
        ? storageDocument?.documentType
        : docType;
    const documentNumber = isValidStorageDocument
        ? storageDocument?.documentNumber
        : '';
    const citizenship = isValidStorageDocument
        ? storageDocument?.citizenship
        : RU_CITIZENSHIP_CODE2;
    const documentValidDate = isValidStorageDocument
        ? storageDocument?.documentValidDate
        : '';

    return {
        lastName: storageDocument?.lastName || '',
        firstName: storageDocument?.firstName || '',
        patronymicName:
            (!middleNameDisabled && storageDocument?.patronymicName) || '',
        birthdate: storageDocument?.birthdate || '',
        sex: storageDocument?.sex,
        documentType,
        documentNumber: documentNumber || '',
        citizenship: citizenship || RU_CITIZENSHIP_CODE2,
        documentValidDate: documentValidDate || '',
        ageCategory,
        isPatronymicDisabled: storageDocument?.isPatronymicDisabled,
        loyaltyCard: getInitialLoyaltyCard(
            storageDocument?.loyaltyCard,
            allowedLoyaltyPrograms,
        ),
    };
}

export const createBookingFormPassengers = (
    passengerSeats: IPassengers,
    storagePassengers: IBookingFormPassengersData[],
    options: ICreateBookingPassengersOptions,
): IBookingFormPassengersData[] => {
    const adultPassengers = storagePassengers.filter(
        passenger => passenger.ageCategory === EAgeCategory.adult,
    );
    const childPassengers = storagePassengers.filter(
        passenger => passenger.ageCategory === EAgeCategory.child,
    );
    const infantPassengers = storagePassengers.filter(
        passenger => passenger.ageCategory === EAgeCategory.infant,
    );

    return [
        ...Array.from(Array(passengerSeats.adults), (_, index) =>
            formStepFactory(
                EAgeCategory.adult,
                adultPassengers[index],
                options,
            ),
        ),
        ...Array.from(Array(passengerSeats.children), (_, index) =>
            formStepFactory(
                EAgeCategory.child,
                childPassengers[index],
                options,
            ),
        ),
        ...Array.from(Array(passengerSeats.infants), (_, index) =>
            formStepFactory(
                EAgeCategory.infant,
                infantPassengers[index],
                options,
            ),
        ),
    ];
};
