import React, {useCallback, useEffect, useMemo, useState} from 'react';
import {Redirect, RouteComponentProps} from 'react-router-dom';
import {connect} from 'react-redux';
import {FormState} from 'final-form';
import {FormSpy} from 'react-final-form';
import get from 'lodash/get';
import last from 'lodash/last';
import first from 'lodash/first';
import countBy from 'lodash/countBy';
import setFieldData from 'final-form-set-field-data';

import {EFormKey} from 'constants/form/EFormKey';
import {MINUTE} from 'utilities/dateUtils/constants';
import {validationInfo} from '../AviaBooking/constants/validation/validationInfo';
import {SOLOMON_AVIA_PAY_BUTTON_CLICK} from 'constants/solomon';

import {IBookingSegmentModel} from 'server/services/AviaBookingService/variants/types';
import {IBookingFormData} from '../AviaBooking/types/IBookingFormData';
import {IBookingFormPassengersData} from '../AviaBooking/types/IBookingFormPassengersData';
import {EAviaGoal} from 'utilities/metrika/types/goals/avia';

import {requestCountries as requestCountriesAction} from 'reducers/common/countries/actions';
import {
    cleanupCreateOrderState,
    createOrderRequest,
} from 'reducers/avia/booking/createOrder/actions';
import {requestVariant as requestVariantAction} from 'reducers/avia/booking/variants/actions';

import {parseDate} from 'utilities/dateUtils';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useUserInfo} from 'utilities/hooks/useUserInfo';
import {HUMAN_DATE_RU, ROBOT} from 'utilities/dateUtils/formats';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {getBookingContacts} from 'utilities/localStorage/booking/getBookingContacts';
import {getBookingPassengers} from 'utilities/localStorage/booking/getBookingPassengers';
import {saveBookingContacts} from 'utilities/localStorage/booking/saveBookingContacts';
import {saveBookingPassengers} from 'utilities/localStorage/booking/saveBookingPassengers';
import {aviaURLs} from 'projects/avia/lib/urls';
import {createBookingFormPassengers} from '../AviaBooking/lib/createBookingFormValues';
import {getFlightType} from '../AviaBooking/lib/getFlightType';
import {sendPaymentCounter} from 'utilities/solomon';
import bookingFormCustomValidation from 'projects/avia/pages/AviaBookingCreateOrder/utilities/bookingFormCustomValidation/bookingFormCustomValidation';
import {reachGoal} from 'utilities/metrika';

import Form from 'components/Form/Form';
import focusFirstInvalidField from 'components/Form/mutators/focusFirstInvalidField';
import createSubmitErrorDecorator from 'components/Form/decorators/createSubmitErrorDecorator';
import BookLoader from 'components/BookLoader/BookLoader';
import {
    BookingBreadcrumbs,
    EBookingBreadcrumbsStep,
} from '../AviaBooking/components/BookingBreadcrumbs/BookingBreadcrumbs';
import {BookingTravelersNotebookContext} from '../AviaBooking/components/BookingTravelersNotebook/BookingTravelersNotebookProvider';
import PortalServerError from 'components/PortalServerError/PortalServerError';
import {
    BookingErrorModal,
    EBookingModalErrorType,
} from '../AviaBooking/components/BookingErrorModal/BookingErrorModal';
import Flex from 'components/Flex/Flex';

import {AviaBookingCreateOrderFormContent} from './AviaBookingCreateOrderFormContent';
import {
    aviaCreateOrderPageSelector,
    IAviaCreateOrderPageData,
} from './aviaCreateOrderPageSelector';
import {AviaBookingOutdatedModal} from './AviaBookingOutdatedModal/AviaBookingOutdatedModal';

const VARIANT_OUTDATED_TIMEOUT = 20 * MINUTE;
const PAGE_QA = 'avia-booking-create-order-page';

type TAviaBookingCreateOrderProps = RouteComponentProps &
    IAviaCreateOrderPageData & {
        createOrder: typeof createOrderRequest;
        requestVariants: typeof requestVariantAction;
        requestCountries: typeof requestCountriesAction;
        cleanup(): void;
    };

const focusOnSubmitErrors = createSubmitErrorDecorator<IBookingFormData>(
    EFormKey.AVIA_BOOK,
);

let submittedToken: string | undefined;

const AviaBookingCreateOrder: React.FunctionComponent<TAviaBookingCreateOrderProps> =
    ({
        variants,
        location,
        orderId,
        travellerNotebook,
        pageHasError,
        orderIsCreating,
        errorOfCreateOrder,
        oldPrices,
        countries,
        createOrder,
        requestVariants,
        requestCountries,
        cleanup,
    }) => {
        const query = getQueryByLocation(location) as Record<string, string>;
        const token = query.token;

        const userInfo = useUserInfo();
        const [priceChangeErrorIsClosed, setPriceChangedErrorIsClosed] =
            useState(false);
        const [sameNameRestrictionError, setSameNameRestrictionError] =
            useState(false);
        const {
            value: isOutdated,
            setValue: setIsOutdated,
            setFalse: resetIsOutdated,
        } = useBoolean(false);

        const initialValues: IBookingFormData | undefined = useMemo(() => {
            if (!variants) {
                return undefined;
            }

            const {allVariants, passengers} = variants;

            const flightInfo = getFlightInfo(allVariants[0].segments);
            const departureDate =
                flightInfo?.lastFlightDepartureDate ||
                flightInfo?.firstFlightDepartureDate ||
                '';
            const initialToken = allVariants.some(v => v.id === token)
                ? token
                : get(first(allVariants), 'id', '');

            const storagePassengers =
                getBookingPassengers<IBookingFormPassengersData>(
                    EFormKey.AVIA_BOOK,
                    userInfo,
                );

            const contacts = getBookingContacts(EFormKey.AVIA_BOOK, userInfo);
            const initialPassengers = createBookingFormPassengers(
                passengers,
                storagePassengers,
                {
                    flightType: getFlightType(allVariants[0].segments),
                    allowedLoyaltyPrograms:
                        allVariants[0].allowedLoyaltyPrograms,
                    middleNameDisabled: false,
                },
            );

            return {
                passengers: initialPassengers,
                contacts,
                flightInfo: {
                    variantToken: initialToken,
                    departureDate: parseDate(departureDate, ROBOT).format(
                        HUMAN_DATE_RU,
                    ),
                },
            };
        }, [variants, token, userInfo]);

        useEffect(() => {
            reachGoal(EAviaGoal.BOY_OPEN);
        }, []);

        useEffect(() => {
            const testTimeOut = Number(query.testOutdateTimeout);
            const handleId = setTimeout(
                () => setIsOutdated(true),
                testTimeOut || VARIANT_OUTDATED_TIMEOUT,
            );

            return (): void => {
                clearTimeout(handleId);
            };
        }, [query.testOutdateTimeout, setIsOutdated]);

        useEffect(() => {
            requestCountries();
        }, [requestCountries]);

        useEffect(() => {
            requestVariants([token]);
        }, [requestVariants, token]);

        useEffect(() => (): void => cleanup(), [cleanup]);

        let errorType: EBookingModalErrorType | undefined;

        if (Boolean(oldPrices[token]) && !priceChangeErrorIsClosed) {
            const variantAvailable = Boolean(
                variants?.allVariants.find(v => v.id === token),
            );

            errorType = variantAvailable
                ? EBookingModalErrorType.PriceChanged
                : EBookingModalErrorType.TariffNotAvailable;
        } else if (sameNameRestrictionError) {
            errorType = EBookingModalErrorType.SameNameRestriction;
        }

        const handleFormStateChange = useCallback(
            (formState: FormState<IBookingFormData>): void => {
                const {submitSucceeded, values} = formState;
                const {passengers = [], contacts} = values;

                if (!submitSucceeded) {
                    saveBookingPassengers<IBookingFormPassengersData>(
                        EFormKey.AVIA_BOOK,
                        passengers,
                        userInfo,
                    );

                    if (contacts) {
                        saveBookingContacts(
                            EFormKey.AVIA_BOOK,
                            contacts,
                            userInfo,
                        );
                    }
                }
            },
            [userInfo],
        );

        const handleOrderCreate = useCallback(
            (formData: IBookingFormData) => {
                const {passengers = []} = formData;

                if (checkEqualNameRestriction(passengers)) {
                    setSameNameRestrictionError(true);
                    reachGoal(EAviaGoal.BOY_PAY_BUTTON_FAIL);

                    return;
                }

                submittedToken = formData.flightInfo?.variantToken;
                reachGoal(EAviaGoal.BOY_PAY_BUTTON);
                createOrder({...formData, marker: query.referrer});
                sendPaymentCounter(SOLOMON_AVIA_PAY_BUTTON_CLICK);
            },
            [createOrder, query.referrer],
        );

        const handleInvalidFormSubmit = useCallback(() => {
            reachGoal(EAviaGoal.BOY_PAY_BUTTON_FAIL);
        }, []);

        const onCloseBookingError = useCallback(() => {
            setSameNameRestrictionError(false);
            setPriceChangedErrorIsClosed(true);
        }, []);

        if (orderId) {
            const paymentUrl = aviaURLs.getBookingPaymentUrl(
                orderId,
                submittedToken || token,
                query,
            );

            return <Redirect push to={paymentUrl} />;
        }

        if (variants) {
            const flightInfo = getFlightInfo(variants.allVariants[0].segments);

            return (
                <BookingTravelersNotebookContext.Provider
                    value={travellerNotebook}
                >
                    <Form<IBookingFormData>
                        initialValues={initialValues}
                        decorators={[focusOnSubmitErrors]}
                        mutators={{focusFirstInvalidField, setFieldData}}
                        onSubmit={handleOrderCreate}
                        onTrySubmit={handleInvalidFormSubmit}
                        validationInfo={validationInfo}
                        validate={bookingFormCustomValidation}
                        subscription={{}}
                        render={({handleSubmit}): React.ReactNode => (
                            <form
                                name={EFormKey.AVIA_BOOK}
                                onSubmit={handleSubmit}
                                autoComplete="off"
                                noValidate
                                {...prepareQaAttributes(PAGE_QA)}
                            >
                                <BookingBreadcrumbs
                                    activeStep={EBookingBreadcrumbsStep.FORM}
                                    variants={variants}
                                    variantToken={token}
                                    {...prepareQaAttributes({
                                        parent: PAGE_QA,
                                        current: 'breadcrumbs',
                                    })}
                                />
                                <AviaBookingCreateOrderFormContent
                                    allVariants={variants.allVariants}
                                    passengers={variants.passengers}
                                    countries={countries}
                                    disabled={orderIsCreating}
                                    error={errorOfCreateOrder}
                                    oldPrices={oldPrices}
                                    {...prepareQaAttributes(PAGE_QA)}
                                />
                                <BookingErrorModal
                                    variantToken={
                                        initialValues?.flightInfo
                                            ?.variantToken || ''
                                    }
                                    variants={variants}
                                    errorType={errorType}
                                    onClose={onCloseBookingError}
                                />
                                <AviaBookingOutdatedModal
                                    isVisible={isOutdated}
                                    qid={variants.qid}
                                    fromTitle={flightInfo.fromCityTitle}
                                    toTitle={flightInfo.toCityTitle}
                                    passengers={variants.passengers}
                                    onClose={resetIsOutdated}
                                />
                                <FormSpy<IBookingFormData>
                                    onChange={handleFormStateChange}
                                    subscription={{
                                        values: true,
                                        submitSucceeded: true,
                                    }}
                                />
                            </form>
                        )}
                    />
                </BookingTravelersNotebookContext.Provider>
            );
        }

        if (pageHasError) {
            return (
                <Flex y={10} justifyContent="center">
                    <PortalServerError />
                </Flex>
            );
        }

        return <BookLoader isLoading />;
    };

interface IFlightInfo {
    firstFlightDepartureDate: string | undefined;
    lastFlightDepartureDate: string | undefined;
    fromCityTitle: string | undefined;
    toCityTitle: string | undefined;
}

export function getFlightInfo(segments: IBookingSegmentModel[]): IFlightInfo {
    return {
        firstFlightDepartureDate: first(first(segments)?.flights)
            ?.departureDate,
        lastFlightDepartureDate: last(last(segments)?.flights)?.departureDate,
        fromCityTitle: first(first(segments)?.flights)?.departureSettlement
            ?.title,
        toCityTitle: last(first(segments)?.flights)?.arrivalSettlement?.title,
    };
}

function checkEqualNameRestriction(
    documents: IBookingFormPassengersData[],
): boolean {
    const MAX_EQUAL_NAME_DOC_COUNT = 3;

    if (documents.length < MAX_EQUAL_NAME_DOC_COUNT) {
        return false;
    }

    const counts = countBy(
        documents,
        ({firstName, lastName}) =>
            `${lastName?.toLowerCase()} ${firstName?.toLowerCase()}`,
    );

    return Object.values(counts).some(
        count => count >= MAX_EQUAL_NAME_DOC_COUNT,
    );
}

const mapDispatchToProps = {
    createOrder: createOrderRequest,
    requestVariants: requestVariantAction,
    requestCountries: requestCountriesAction,
    cleanup: cleanupCreateOrderState,
};

export default connect(
    aviaCreateOrderPageSelector,
    mapDispatchToProps,
)(AviaBookingCreateOrder);
