import React, {useRef, useContext, useMemo} from 'react';
import Foco from 'react-foco';
import {useSelector} from 'react-redux';

import {ICountry} from 'types/common/ICountry';
import {IBookingVariantModel} from 'server/services/AviaBookingService/variants/types';
import {IPassengers} from '../AviaBooking/types/IPassengers';
import {EFlightInfoFieldName} from 'projects/avia/pages/AviaBooking/types/IBookingFlightInfo';
import {EFormGroup} from 'projects/avia/pages/AviaBooking/types/IBookingFormData';

import experimentsSelector from 'selectors/common/experimentsSelector';

import IPrice from 'utilities/currency/PriceInterface';
import {getFlightType} from '../AviaBooking/lib/getFlightType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {mapPassengersToTypes} from '../AviaBooking/components/AviaBookingForm/utilities/mapPassengersToTypes';
import {groupPassengersByType} from '../AviaBooking/components/AviaBookingForm/utilities/groupPassengersByType';
import {useBoolean} from 'utilities/hooks/useBoolean';

import * as i18nBlock from 'i18n/avia-AviaBooking-BookingPageContent';
import * as i18nBlockAviaPlus from 'i18n/avia-plusPromo2021';

import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import Box from 'components/Box/Box';
import Button from 'components/Button/Button';
import Separator from 'components/Separator/Separator';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';
import {AviaBookingDisclaimer} from 'projects/avia/components/AviaBookingDisclaimer/AviaBookingDisclaimer';
import {BookingTravelersNotebookContext} from '../AviaBooking/components/BookingTravelersNotebook/BookingTravelersNotebookProvider';
import AviaBookingForm from '../AviaBooking/components/AviaBookingForm/AviaBookingForm';
import BookingSegmentsInfo from '../AviaBooking/components/BookingSegmentsInfo/BookingSegmentsInfo';
import TariffsField from '../AviaBooking/components/TariffsField/TariffsField';
import {BookingPriceInfo} from '../AviaBooking/components/BookingPriceInfo/BookingPriceInfo';
import SupportPhone from '../AviaBooking/components/SupportPhone/SupportPhone';
import {useFieldValue} from 'components/Form/hooks/useFieldValue';
import PlusInfo from 'components/PlusInfo/PlusInfo';

import DeviceTypeContext from 'contexts/DeviceTypeContext';

import cx from './AviaBookingCreateOrderFormContent.scss';

interface IAviaBookingCreateOrderFormContentProps extends IWithQaAttributes {
    allVariants: IBookingVariantModel[];
    passengers: IPassengers;
    countries: ICountry[];
    disabled?: boolean;
    error?: Error | null;
    oldPrices: Record<string, IPrice | undefined>;
}

export const AviaBookingCreateOrderFormContent: React.FC<IAviaBookingCreateOrderFormContentProps> =
    props => {
        const {
            allVariants,
            passengers,
            countries,
            disabled,
            error = true,
            oldPrices,
        } = props;
        const deviceType = useContext(DeviceTypeContext);
        const {isMobile} = deviceType;
        const submitButtonRef = useRef<HTMLElement | null>(null);
        const {
            value: submitErrorIsClosed,
            setTrue: closeSubmitError,
            setFalse: showSubmitError,
        } = useBoolean(false);
        const experiments = useSelector(experimentsSelector);
        const {aviaAeroflotPlus} = experiments;
        const types = useMemo(
            () => mapPassengersToTypes(passengers),
            [passengers],
        );
        const notebook = useContext(BookingTravelersNotebookContext);

        const variantToken =
            useFieldValue<string>(
                EFlightInfoFieldName.variantToken,
                EFormGroup.flightInfo,
            ) || '';

        const activeVariant = allVariants.find(
            variant => variantToken === variant.id,
        );

        const plusPoints = allVariants.some(
            variant => variant?.promoCampaigns?.plusPromo2021?.enabled,
        )
            ? activeVariant?.promoCampaigns?.plusPromo2021?.totalPlusPoints || 0
            : null;

        const passengersGroupedByType = useMemo(
            () =>
                groupPassengersByType({
                    types,
                    segments: activeVariant ? activeVariant.segments : [],
                    passengersWithDocuments:
                        notebook?.passengersWithDocuments || [],
                }),
            [notebook, activeVariant, types],
        );

        const renderRightColumn = (): React.ReactNode => {
            return (
                <div className={cx('rightColumn')}>
                    <BookingLayout.Cart>
                        <Box between={isMobile ? 4 : 5}>
                            <TariffsField
                                variants={allVariants}
                                disabled={disabled}
                                oldPrices={oldPrices}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'tariffs',
                                })}
                            />
                            <Separator />
                            <div>
                                {activeVariant && (
                                    <BookingPriceInfo
                                        priceInfo={activeVariant.priceInfo}
                                        oldPrice={oldPrices[variantToken]}
                                        {...prepareQaAttributes({
                                            parent: props,
                                            current: 'price-info',
                                        })}
                                    />
                                )}
                                {aviaAeroflotPlus &&
                                    activeVariant &&
                                    typeof plusPoints === 'number' && (
                                        <PlusInfo
                                            className={cx('plusInfo')}
                                            plusPoints={plusPoints}
                                            popupText={i18nBlockAviaPlus.plusPopupBookingText()}
                                            popupTitle={i18nBlockAviaPlus.plusTitle()}
                                        />
                                    )}
                            </div>
                            <MessageBoxPopup
                                anchorRef={submitButtonRef}
                                isVisible={Boolean(
                                    error && !submitErrorIsClosed,
                                )}
                            >
                                <div
                                    className={cx('submitErrorTooltipContent')}
                                >
                                    {i18nBlock.submitDashError()}
                                </div>
                            </MessageBoxPopup>
                            <Foco
                                onClickOutside={closeSubmitError}
                                onFocusOutside={closeSubmitError}
                            >
                                <Button
                                    size={isMobile ? 'l' : 'xl'}
                                    width="max"
                                    theme="primary"
                                    type="submit"
                                    disabled={disabled}
                                    onClick={showSubmitError}
                                    innerRef={submitButtonRef}
                                    {...prepareQaAttributes({
                                        parent: props,
                                        current: 'submit',
                                    })}
                                >
                                    {disabled
                                        ? i18nBlock.submittingDashTitle()
                                        : i18nBlock.submitDashTitle()}
                                </Button>
                            </Foco>
                        </Box>
                    </BookingLayout.Cart>
                    <BookingLayout.CartCaption className={cx('disclaimer')}>
                        <AviaBookingDisclaimer />
                    </BookingLayout.CartCaption>
                </div>
            );
        };

        const leftColumn = activeVariant && (
            <div className={cx('leftColumn')}>
                {isMobile && <SupportPhone className={cx('supportPhone')} />}
                <BookingSegmentsInfo
                    {...prepareQaAttributes(props)}
                    segments={activeVariant.segments}
                    withBaggage
                />

                <AviaBookingForm
                    countries={countries}
                    disabled={disabled}
                    flightType={getFlightType(activeVariant.segments)}
                    allowedLoyaltyPrograms={
                        activeVariant.allowedLoyaltyPrograms
                    }
                    types={types}
                    passengersGroupedByType={passengersGroupedByType}
                    {...prepareQaAttributes({parent: props, current: 'form'})}
                />
            </div>
        );

        return (
            <BookingLayout
                deviceType={deviceType}
                leftColumn={leftColumn}
                rightColumn={renderRightColumn()}
                className={cx('bookingLayout')}
            />
        );
    };
