import React, {useMemo} from 'react';

import {EAppActions} from 'constants/platforms/TPlatforms';

import {IPassengers} from '../../AviaBooking/types/IPassengers';

import {aviaURLs} from 'projects/avia/lib/urls';
import {qidToSearchForm} from 'projects/avia/lib/qid';

import * as i18nBlock from 'i18n/avia-AviaBooking-error-modals';

import ErrorModal, {TErrorActionType} from 'components/ErrorModal/ErrorModal';
import Flex from 'components/Flex/Flex';
import {SearchInfo} from './components/SearchInfo/SearchInfo';

import {useCoordinator, usePlatform} from 'contexts/PlatformContext';

import cx from './AviaBookingOutdatedModal.scss';

interface IAviaBookingOutdatedModalProps {
    qid: string;
    isVisible: boolean;
    fromTitle?: string;
    toTitle?: string;
    passengers: IPassengers;
    onClose(): void;
}

export const AviaBookingOutdatedModal: React.FC<IAviaBookingOutdatedModalProps> =
    ({fromTitle, isVisible, passengers, qid, toTitle, onClose}) => {
        const searchForm = useMemo(() => {
            return qidToSearchForm(qid);
        }, [qid]);
        const {isTravelApp} = usePlatform();
        const coordinator = useCoordinator();

        const text = useMemo(() => {
            return (
                <Flex
                    className={cx('text')}
                    flexDirection="column"
                    between={4}
                    alignItems="stretch"
                >
                    <div>{i18nBlock.outdatedDashDescription()}</div>
                    <SearchInfo
                        fromTitle={fromTitle}
                        toTitle={toTitle}
                        fromDate={searchForm.when}
                        toDate={searchForm.return_date}
                        passengers={passengers}
                    />
                </Flex>
            );
        }, [fromTitle, toTitle, searchForm, passengers]);

        const primaryAction: TErrorActionType = useMemo(() => {
            const title = i18nBlock.outdatedDashButtonDashRetry();

            if (isTravelApp) {
                return {
                    type: 'button',
                    title,
                    handler: (): void => {
                        coordinator.doAction(
                            EAppActions.NAVIGATE_TO_SEARCH_AVIA_PAGE,
                        );
                    },
                };
            }

            return {
                type: 'button-link',
                title,
                props: {
                    to: aviaURLs.getSearchResultsUrl(searchForm),
                },
            };
        }, [coordinator, isTravelApp, searchForm]);

        const secondaryAction: TErrorActionType = useMemo(
            () => ({
                type: 'button',
                title: i18nBlock.priceDashChangeDashButtonDashContinue(),
                handler: onClose,
            }),
            [onClose],
        );

        return (
            <ErrorModal
                title={i18nBlock.outdatedDashTitle()}
                text={text}
                isVisible={isVisible}
                primaryAction={primaryAction}
                secondaryAction={secondaryAction}
            />
        );
    };
