import {createSelector} from 'reselect';
import {RouteComponentProps} from 'react-router-dom';

import {ICountry} from 'types/common/ICountry';
import {ITravelerNotebook} from 'projects/avia/pages/AviaBooking/types/ITravelerNotebook';
import {IBookingVariantsModelType} from 'server/services/AviaBookingService/variants/types';

import {StoreInterface} from 'reducers/storeTypes';
import {AviaBookingVariantsResource} from 'reducers/avia/booking/variants/reducer';

import {
    getAviaBookingVariant,
    getAviaTravellerNotebook,
    getCreateOrderState,
} from 'selectors/avia/aviaBooking';
import {oldPricesSelector} from 'selectors/avia/booking/oldPricesSelector';
import {getAllCountries} from 'selectors/common/countriesSelectors';

import IPrice from 'utilities/currency/PriceInterface';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';

const getAviaBookingVariantSelector = (
    state: StoreInterface,
    props: RouteComponentProps,
): AviaBookingVariantsResource => {
    const {token} = getQueryByLocation(props.location) as Record<
        string,
        string
    >;

    return getAviaBookingVariant(state, token);
};

export interface IAviaCreateOrderPageData {
    variants: IBookingVariantsModelType | null;
    countries: ICountry[];
    pageHasError: boolean;
    pageIsLoading: boolean;
    travellerNotebook: ITravelerNotebook | null;
    orderId: string | null;
    orderIsCreating: boolean;
    errorOfCreateOrder: Error | null;
    oldPrices: Record<string, IPrice | undefined>;
}

export const aviaCreateOrderPageSelector = createSelector(
    [
        getAviaBookingVariantSelector,
        getAviaTravellerNotebook,
        getCreateOrderState,
        oldPricesSelector,
        getAllCountries,
    ],
    (
        variantState,
        {travellerNotebook, loading: travellerNotebookIsLoading},
        {error: errorOfCreateOrder, loading: orderIsCreating, orderId},
        oldPrices,
        countries,
    ): IAviaCreateOrderPageData => {
        const pageIsLoading =
            !variantState || variantState.loading || travellerNotebookIsLoading;

        const pageHasError = Boolean(variantState && variantState.error);

        const variants = variantState && variantState.variants;

        return {
            variants,
            pageHasError,
            pageIsLoading,
            travellerNotebook,
            orderId,
            orderIsCreating,
            errorOfCreateOrder,
            oldPrices,
            countries,
        };
    },
);
