import React from 'react';

import {ITravellerInfoDTO} from 'server/api/AviaBookingApi/types/ITravellerInfoDTO';

import {useMobile} from 'utilities/hooks/useMobile';
import {formatDate} from 'utilities/dateUtils';
import {HUMAN_DATE_RU} from 'utilities/dateUtils/formats';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {getDocumentTitle} from 'projects/avia/pages/AviaBooking/lib/documentUtils';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBlock from 'i18n/avia-AviaBookingOrder-BookingTraveller';

import Box from 'components/Box/Box';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';

interface IBookingTravellerProps extends IWithQaAttributes {
    traveller: ITravellerInfoDTO;
    withPrices?: boolean;
}

export const BookingTraveller: React.FC<IBookingTravellerProps> = props => {
    const {traveller, withPrices} = props;
    const isMobile = useMobile();

    return (
        <Box {...prepareQaAttributes(props)}>
            <Text
                {...prepareQaAttributes({
                    parent: props,
                    current: 'fullName',
                })}
                size={isMobile ? 'm' : 'l'}
                weight="medium"
            >
                {i18nBlock
                    .fullDashName({
                        lastName: traveller.lastName || '',
                        firstName: traveller.firstName || '',
                        middleName: traveller.middleName || '',
                    })
                    .trim()}
            </Text>
            <Box
                {...prepareQaAttributes({
                    parent: props,
                    current: 'fullDocument',
                })}
                textSize={isMobile ? 'm' : 's'}
            >
                {i18nBlock.document({
                    title: getDocumentTitle(traveller.documentType),
                    number: (traveller.documentNumber || '').toUpperCase(),
                    validTo: traveller.documentValidTill
                        ? formatDate(traveller.documentValidTill, HUMAN_DATE_RU)
                        : false,
                })}
            </Box>
            <Box
                {...prepareQaAttributes({
                    parent: props,
                    current: 'birthDate',
                })}
                textSize={isMobile ? 'm' : 's'}
                below="2"
            >
                {formatDate(traveller.dateOfBirth, HUMAN_DATE_RU)}
            </Box>
            {withPrices && (
                <Box
                    below="2"
                    textSize={isMobile ? 'm' : 's'}
                    textColor="secondary"
                >
                    {insertJSXIntoKey(i18nBlock.totalDashCost)({
                        price: (
                            <Price
                                key="total-cost"
                                value={traveller.priceInfo.total.value}
                                currency={traveller.priceInfo.total.currency}
                            />
                        ),
                    })}
                </Box>
            )}
        </Box>
    );
};
