import React from 'react';

import {ITravellerInfoDTO} from 'server/api/AviaBookingApi/types/ITravellerInfoDTO';

import IPrice from 'utilities/currency/PriceInterface';
import {useMobile} from 'utilities/hooks/useMobile';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBlock from 'i18n/avia-AviaBookingOrder-BookingTravellerList';

import Card from 'components/Card/Card';
import Intersperse from 'components/Intersperse/Intersperse';
import Heading from 'components/Heading/Heading';
import Price from 'components/Price/Price';
import Text from 'components/Text/Text';
import Box from 'components/Box/Box';
import Separator from 'components/Separator/Separator';
import {BookingCardContent} from 'projects/avia/pages/AviaBooking/components/BookingCardContent/BookingCardContent';
import {BookingTraveller} from 'projects/avia/pages/AviaBookingOrder/components/BookingTraveller';

import cx from './BookingTravellerList.scss';

interface IBookingTravellerListProps extends IWithQaAttributes {
    travellers: ITravellerInfoDTO[];
    price?: IPrice;
    withPrices?: boolean;
}

export const BookingTravellerList: React.FC<IBookingTravellerListProps> =
    props => {
        const {price, travellers, withPrices = true} = props;
        const isMobile = useMobile();

        return (
            <Card {...prepareQaAttributes(props)} shadow="default">
                <Intersperse separator={<Separator />}>
                    <BookingCardContent>
                        <Heading level={isMobile ? 3 : 2}>
                            {i18nBlock.title()}
                        </Heading>
                    </BookingCardContent>

                    {travellers.map(traveller => (
                        <BookingCardContent key={traveller.id}>
                            <BookingTraveller
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'traveller',
                                    key: traveller.id,
                                })}
                                traveller={traveller}
                                withPrices={withPrices}
                            />
                        </BookingCardContent>
                    ))}

                    {withPrices && (
                        <BookingCardContent>
                            <Box below={2}>
                                <Text
                                    size={isMobile ? 'm' : 'l'}
                                    weight={isMobile ? 'bold' : 'medium'}
                                    className={cx({wide: isMobile})}
                                >
                                    {insertJSXIntoKey(
                                        isMobile
                                            ? i18nBlock.totalDashPriceDashMobile
                                            : i18nBlock.totalDashPrice,
                                    )({
                                        price: price && (
                                            <Price
                                                key="price"
                                                value={price.value}
                                                currency={price.currency}
                                                {...prepareQaAttributes({
                                                    parent: props,
                                                    current: 'price',
                                                })}
                                            />
                                        ),
                                    })}
                                </Text>
                            </Box>
                        </BookingCardContent>
                    )}
                </Intersperse>
            </Card>
        );
    };
