import React, {useMemo, useEffect} from 'react';
import {RouteComponentProps} from 'react-router-dom';
import {useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';

import EGenericOrderSource from 'server/api/GenericOrderApi/types/common/EGenericOrderSource';
import {EFooterProject} from 'components/Footer/types';
import EHeaderBorderBottomType from 'components/Header/types/EHeaderBorderBottomType';

import {getOrderAuthorization} from 'selectors/common/checkOrderAuthorizationSelector';

import {useAsync} from 'utilities/hooks/useAsync';
import {checkOrderAccess} from 'projects/account/utilities/order/checkOrderAccess';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import CheckOrderAuthorizationContainer from 'components/CheckOrderAuthorization/CheckOrderAuthorizationContainer';
import PortalServerError from 'components/PortalServerError/PortalServerError';
import LayoutDefault, {
    ELayoutBgColor,
} from 'components/Layouts/LayoutDefault/LayoutDefault';
import BookLoader from 'components/BookLoader/BookLoader';
import {AviaBookingOrderPageView} from 'projects/avia/pages/AviaBookingOrder/components/AviaBookingOrderPageView';
import {Center} from 'projects/avia/components/Center/Center';

import {bookingApiClient} from 'projects/avia/pages/AviaBooking/api/bookingApiClient';

import {usePlatform} from 'contexts/PlatformContext';

import cx from './AviaBookingOrderPage.scss';

interface IAviaBookingOrderParams {
    id: string;
}

interface IAviaBookingOrderPageProps
    extends RouteComponentProps<IAviaBookingOrderParams> {}

const AviaBookingOrderPage: React.FC<IAviaBookingOrderPageProps> = props => {
    const {id} = props.match.params;

    const orderAuthorization = useSelector(getOrderAuthorization);
    const hasAccess = checkOrderAccess(id, orderAuthorization);
    const [order, fetchOrder] = useAsync(bookingApiClient.getOrder);
    const deviceType = useDeviceType();
    const {isWeb} = usePlatform();

    useEffect(() => {
        if (hasAccess) {
            fetchOrder(id, EGenericOrderSource.ORDER_PAGE);
        }
    }, [id, hasAccess, fetchOrder]);

    const content = useMemo(() => {
        if (!hasAccess) {
            return (
                <Center y={5} x={4} vertical={false}>
                    <CheckOrderAuthorizationContainer orderId={id} />
                </Center>
            );
        }

        if (order.loading) {
            return <BookLoader isLoading />;
        }

        if (order.error) {
            return (
                <Center y={5} x={4} vertical={false}>
                    <PortalServerError />
                </Center>
            );
        }

        return <AviaBookingOrderPageView order={order.data} />;
    }, [hasAccess, id, order.data, order.error, order.loading]);

    return (
        <LayoutDefault
            className={cx('root', deviceMods('root', deviceType))}
            footerClassName={cx('footer')}
            showNavigation={deviceType.isDesktop}
            showHeader={isWeb}
            showFooter={isWeb}
            bgColor={
                deviceType.isMobile ? ELayoutBgColor.DARK : ELayoutBgColor.LIGHT
            }
            searchFormInitialIsExpanded={false}
            headerBorderBottomType={EHeaderBorderBottomType.NONE}
            footerIncludeSupportBlock
            project={EProjectName.ACCOUNT}
            footerType={EFooterProject.ACCOUNT}
        >
            <div className={cx('content')}>{content}</div>
        </LayoutDefault>
    );
};

export default React.memo(AviaBookingOrderPage);
