import React, {useEffect, useState} from 'react';
import {Redirect, useLocation} from 'react-router-dom';

import {useMobile} from 'utilities/hooks/useMobile';
import {useBoolean} from 'utilities/hooks/useBoolean';
import IPrice from 'utilities/currency/PriceInterface';
import {areEqualPrices} from 'utilities/currency/areEqualPrices';
import {
    isFinalOrderState,
    isErrorOrderState,
} from '../AviaBooking/lib/orderUtils';
import {aviaURLs} from 'projects/avia/lib/urls';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {useExperiments} from 'utilities/hooks/useExperiments';

import CheckOrderAuthorizationContainer from 'components/CheckOrderAuthorization/CheckOrderAuthorizationContainer';
import BookLoader from 'components/BookLoader/BookLoader';
import {Center} from 'projects/avia/components/Center/Center';
import {
    BookingBreadcrumbs,
    EBookingBreadcrumbsStep,
} from '../AviaBooking/components/BookingBreadcrumbs/BookingBreadcrumbs';
import {
    BookingErrorModal,
    EBookingModalErrorType,
} from '../AviaBooking/components/BookingErrorModal/BookingErrorModal';
import {BookingPaymentContent} from './components/BookingPaymentContent/BookingPaymentContent';

import {EAeroflotOrderState} from 'server/api/AviaBookingApi/enums/EAeroflotOrderState';

import {
    IBookingPaymentDispatchProps,
    IBookingPaymentOwnProps,
    IBookingPaymentStateProps,
} from './BookingPaymentContainer';

import cx from './BookingPayment.scss';

interface IBookingPaymentProps
    extends IBookingPaymentOwnProps,
        IBookingPaymentStateProps,
        IBookingPaymentDispatchProps {}

export const BookingPayment: React.FC<IBookingPaymentProps> = ({
    compositeState,
    match,
    variants,
    initPayment,
    orderResource,
    paymentIsFailed,
    requestVariants,
    startStatePolling,
    stopStatePolling,
    requestOrder,
}) => {
    const {orderId, token} = match.params;

    const isMobile = useMobile();

    const [newPrice, setNewPrice] = useState<IPrice | null>(null);
    const {value: modalIsClosedByUser, setTrue: closeModal} = useBoolean(false);
    const {value: hasAccess, setTrue: grantAccess} = useBoolean(false);
    const {value: priceIsVerified, setTrue: setPriceIsVerified} =
        useBoolean(false);

    const hasVariants = Boolean(variants);

    const location = useLocation();
    const {aviaNewTrustForm} = useExperiments();

    const invoiceState = compositeState?.invoiceState;
    const orderState = compositeState?.orderState;
    const order = orderResource?.order;

    useEffect(() => {
        if (!hasAccess) {
            return undefined;
        }

        startStatePolling({orderId});

        return (): void => {
            stopStatePolling();
        };
    }, [orderId, startStatePolling, stopStatePolling, hasAccess]);

    // Данные о вариантах нужны для формирования хлебных крошек и для того,
    // чтобы отслеживать изменение цены
    useEffect(() => {
        if (hasAccess && !hasVariants) {
            requestVariants([token]);
        }
    }, [hasVariants, requestVariants, token, hasAccess]);

    // При переходе статуса заказа в OS_WAIT_CARD_TOKENIZED цена уже утвердилась
    // Утверждённая цена может отличать от той, которая была изначально
    // во время создания заказа.
    // Необходимо запросить данные о заказе, для того чтобы узнать новую цену
    useEffect(() => {
        if (orderState === EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED) {
            requestOrder([orderId]);
        }
    }, [orderId, orderState, requestOrder]);

    // При загрузке данных заказа, необходимо сверить цену заказа
    // Если она изменилась, то нужно уведомить пользователя об этом
    useEffect(() => {
        if (order) {
            const {preliminaryPrice, price} = order;

            if (!areEqualPrices(preliminaryPrice, price)) {
                setNewPrice(price);
            }

            setPriceIsVerified();
        }
    }, [order, setPriceIsVerified]);

    // Если invoiceState отсутствует, то это сигнализирует о том
    // что оплата не была проинициализирована.
    // Оплату заказа можно начинать только после статуса OS_WAIT_CARD_TOKENIZED
    useEffect(() => {
        if (
            !invoiceState &&
            orderState === EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED
        ) {
            initPayment({orderId, isMobile});
        }
    }, [invoiceState, initPayment, isMobile, orderId, orderState]);

    if (!hasAccess) {
        return (
            <>
                <Center y="10" x="4">
                    <CheckOrderAuthorizationContainer
                        orderId={orderId}
                        onSuccessOrderAuthorization={grantAccess}
                    />
                </Center>
            </>
        );
    }

    if (isFinalOrderState(orderState)) {
        const query = getQueryByLocation(location) as Record<string, string>;

        return <Redirect to={aviaURLs.getBookingResultUrl(orderId, query)} />;
    }

    let errorType;

    if (isErrorOrderState(orderState) || paymentIsFailed) {
        // Показываем ошибку с невозможность оплаты на нашем сервисе
        // предлагаем пользователю продолжить оплату на странице партнёра
        errorType = EBookingModalErrorType.Unknown;
    } else if (newPrice && !modalIsClosedByUser) {
        // Указываем пользователю, что цена изменилась
        errorType = EBookingModalErrorType.PriceChanged;
    }

    if (!variants || !compositeState) {
        return (
            <>
                <BookLoader isLoading />
                <BookingErrorModal
                    variantToken={token}
                    variants={variants}
                    errorType={errorType}
                    newPrice={newPrice}
                    onClose={closeModal}
                />
            </>
        );
    }

    const {paymentUrl, confirmationUrl} = compositeState;

    return (
        <>
            {!aviaNewTrustForm && (
                <div className={cx({breadcrumbs: !isMobile})}>
                    <BookingBreadcrumbs
                        activeStep={EBookingBreadcrumbsStep.PAYMENT}
                        variants={variants}
                        variantToken={token}
                        {...prepareQaAttributes('avia-booking-breadcrumbs')}
                    />
                </div>
            )}
            <BookingPaymentContent
                priceIsVerified={
                    errorType !== EBookingModalErrorType.PriceChanged &&
                    priceIsVerified
                }
                confirmationUrl={confirmationUrl}
                paymentUrl={paymentUrl}
                orderId={orderId}
                order={order}
            />
            <BookingErrorModal
                variantToken={token}
                variants={variants}
                errorType={errorType}
                newPrice={newPrice}
                onClose={closeModal}
            />
        </>
    );
};
