import React from 'react';

import {IWithClassName} from 'types/withClassName';
import {ESubscriptionSource} from 'types/subscription/ESubscriptionSource';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';
import {ESubscriptionCode} from 'types/subscription/ESubscriptionCode';

import {useExperiments} from 'utilities/hooks/useExperiments';
import {IWithQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nSpinnerBlock from 'i18n/avia-AviaBooking-BookingTrustLikeSpinner';

import BookingMailSubscription from 'containers/BookingMailSubscription/BookingMailSubscription';

import NewPaymentLayout from 'components/Layouts/NewPaymentLayout/PaymentLayout';
import PaymentLayout from 'components/Layouts/PaymentLayout/PaymentLayout';
import PaymentHeader from 'components/Layouts/PaymentLayout/components/PaymentHeader/PaymentHeader';
import BookingPaymentInfo from 'projects/avia/pages/AviaBookingPayment/components/BookingPaymentInfo/BookingPaymentInfo';
import PaymentOrderInfo from 'projects/avia/pages/AviaBookingPayment/components/PaymentOrderInfo';

import {IAviaBookingOrderPageData} from 'server/services/AviaBookingService/order/IAviaBookingOrderPageData';

export interface IAviaPaymentLayout extends IWithClassName, IWithQaAttributes {
    order: IAviaBookingOrderPageData | null | undefined;
    /**
     * Участвует в логике показа кнопки "подписаться" под формой оплаты.
     * После начала оплаты пользователь будет подписан, если не отжал галку.
     */
    paymentIsStarted: boolean;

    loading?: boolean;
    spinnerText?: string;
    /**
     * @default true
     */
    showFooter?: boolean;
    /**
     * @default true
     */
    showOrderInfo?: boolean;
    children?: React.ReactNode;
}

/**
 * Компонент-обертка для лэйаута оплаты авиа. Нужен на время проведения эксперимента
 * по новой форме оплаты траста. Умеет показывать как старую, так и новую форму
 * оплаты траста и обвязку компонентов вокруг нее.
 */
const AviaPaymentLayout: React.FC<IAviaPaymentLayout> = props => {
    const {
        order,
        paymentIsStarted,
        loading,
        spinnerText = i18nSpinnerBlock.waitDashMessage(),
        showFooter = true,
        showOrderInfo = true,
        children,
        ...rest
    } = props;

    const {aviaNewTrustForm} = useExperiments();

    const PaymentLayoutComponent = aviaNewTrustForm
        ? NewPaymentLayout
        : PaymentLayout;

    return (
        <PaymentLayoutComponent
            {...rest}
            header={
                aviaNewTrustForm ? undefined : (
                    <PaymentHeader
                        noLogo
                        withSkeleton
                        orderNumber={order?.prettyOrderId}
                    />
                )
            }
            footer={
                showFooter && order ? (
                    <BookingMailSubscription
                        email={order.email}
                        source={ESubscriptionSource.PAYMENT}
                        travelVerticalName={ESubscriptionVerticalName.Avia}
                        promoSubscriptionCode={ESubscriptionCode.travelNews}
                        subscriptionIsAvailable={paymentIsStarted}
                    />
                ) : null
            }
            orderInfo={
                showOrderInfo &&
                (aviaNewTrustForm && order ? (
                    <PaymentOrderInfo order={order} />
                ) : (
                    <BookingPaymentInfo order={order} />
                ))
            }
            loading={loading}
            spinnerText={spinnerText}
        >
            {children}
        </PaymentLayoutComponent>
    );
};

export default React.memo(AviaPaymentLayout);
