import React, {useEffect, useRef, useState} from 'react';

import {useMobile} from 'utilities/hooks/useMobile';
import postMessageEvents from 'projects/avia/lib/postMessageEvents';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import IRedirectTargetLoadedMessage, {
    isRedirectTargetLoadedMessage,
} from 'projects/avia/pages/AviaBookingPaymentRedirectTarget/utilities/IRedirectTargetLoadedMessage';

import SecureIFrameProxy from 'components/SecureIFrameProxy/SecureIFrameProxy';
import AviaPaymentLayout from 'projects/avia/pages/AviaBookingPayment/components/AviaPaymentLayout/AviaPaymentLayout';

import {IAviaBookingOrderPageData} from 'server/services/AviaBookingService/order/IAviaBookingOrderPageData';

import bookingPaymentCx from '../../BookingPayment.scss';

interface IBookingPaymentConfirmationScreenProps {
    confirmationIsDone: boolean;
    confirmationUrl?: string;
    order?: IAviaBookingOrderPageData | null;
    onDone?: (message: IRedirectTargetLoadedMessage) => void;
}

/**
 * Подтверждение оплаты на стороне партнера (Аэрофлота)
 */
export const BookingPaymentConfirmationScreen: React.FC<IBookingPaymentConfirmationScreenProps> =
    ({confirmationIsDone, confirmationUrl, order, onDone}) => {
        const isMobile = useMobile();
        const frameRef = useRef<HTMLIFrameElement>(null);
        const [iframeReady, setReady] = useState(false);

        useEffect(() => {
            return postMessageEvents.subscribe((event: MessageEvent) => {
                const {data: message} = event;

                if (isRedirectTargetLoadedMessage(message)) {
                    onDone?.(message);
                }
            });
        }, [onDone]);

        useEffect(() => {
            if (iframeReady && frameRef.current) {
                try {
                    frameRef.current.scrollIntoView();
                } catch (e) {}
            }
        }, [frameRef, iframeReady]);

        return (
            <AviaPaymentLayout
                order={order}
                paymentIsStarted={true}
                loading={!iframeReady || confirmationIsDone}
                showFooter={false}
            >
                <SecureIFrameProxy
                    src={confirmationUrl}
                    frameRef={frameRef}
                    {...prepareQaAttributes(
                        'avia-booking-payment-confirmation-frame',
                    )}
                    className={bookingPaymentCx(
                        'iframe',
                        'iframe_confirmation',
                        {
                            iframe_mobile: isMobile,
                            iframe_desktop: !isMobile,
                        },
                    )}
                    onLoad={(): void => {
                        setReady(true);
                    }}
                />
            </AviaPaymentLayout>
        );
    };
